'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

exports.default = function (program) {

  /**
   * Check if the license is in the "old" format
   * @param {KbnServer} server A kbnServer.server instance
   *
   * @returns {boolean} license is/is not in the new format
   */
  async function checkLicense(server, options) {
    const licenseCheck = (0, _license_format_check2.default)(server);
    const resp = await licenseCheck.checkLicenseFormat();
    if (!resp) {
      const investigateVersion = server.config().get('pkg.kibiVersion');
      const leftPad = ''.padStart(4, ' ');
      process.stdout.write('\n');
      process.stdout.write(leftPad + `Siren Platform ${investigateVersion} requires a new license format.\n`);
      process.stdout.write(leftPad + 'The installed license is of the older format and should be replaced.\n');
      process.stdout.write(leftPad + 'To prevent reduced functionality, please request an updated license on support.siren.io\n');
      process.stdout.write(leftPad + 'or contact your customer representative before proceeding.\n\n');
      // if -y option passed, skip asking the question about continuing and just continue.
      if (options.Y) {
        return true;
      }
      const rl = _readlinePromise2.default.createInterface(process.stdin, process.stdout);
      const response = await rl.questionAsync(leftPad + 'Continue with incompatible license? [N/y]');
      return ['y', 'Y'].includes(response.toLowerCase().trim()[0]);
    }
  };

  /**
   * Check if the security is enabled, if it is then displays a warning
   * @param {object} config Config object
   *
   * @returns {boolean} continue to upgrade
   */
  async function checkSecurity(config, options) {
    if (config.investigate_access_control && config.investigate_access_control.enabled === true) {
      const leftPad = ''.padStart(4, ' ');
      process.stdout.write('\n');
      process.stdout.write(leftPad + 'Please refer to the breaking changes documentation and' + ' apply any permission changes needed before continuing. Failing to do this can result in an upgrade failure.\n');
      // if -y option passed, skip asking the question about continuing and just continue.
      if (options.Y) {
        return true;
      }
      const rl = _readlinePromise2.default.createInterface(process.stdin, process.stdout);
      const response = await rl.questionAsync(leftPad + 'Do you want to continue without changes? [N/y]');
      return ['y', 'Y'].includes(response.toLowerCase().trim()[0]);
    }
  };

  /**
   * Waits for the kbnServer status to be green.
   *
   * @param {KbnServer} kbnServer A KbnServer instance.
   * @param {Number} retries The number of retries.
   */
  async function waitForGreenStatus(kbnServer, retries) {
    if (retries === 0) {
      throw new Error('Timed out while waiting for the server status to be ' + 'green, please check the logs and try again.');
    }
    if (!kbnServer.status.isGreen()) {
      await _bluebird2.default.delay(2500);
      await waitForGreenStatus(kbnServer, --retries);
    }
  }

  async function runUpgrade(options, config) {
    const configFiles = [options.config];
    if (options.dev && (0, _validate_config.checkConfigYmlExists)('investigate', true)) {
      configFiles.push((0, _validate_config.getConfigYmlPath)('investigate', true));
    }

    (0, _lodash.merge)(config, {
      env: 'production',
      logging: {
        silent: false,
        quiet: false,
        verbose: false,
        dest: 'stdout'
      },
      optimize: {
        enabled: false
      },
      server: {
        autoListen: false
      },
      plugins: {
        initialize: true,
        scanDirs: options.pluginDir
      },
      migrations: {
        enabled: false
      },
      gremlin: {
        enabled: false
      },
      configFiles: configFiles
    });

    const kbnServer = new _kbn_server2.default(config);

    await kbnServer.ready();

    const logger = new _migration_logger2.default(kbnServer.server, 'migrations');
    const runner = new _migration_runner2.default(kbnServer.server, logger, kbnServer);

    try {
      await waitForGreenStatus(kbnServer, 10);

      const licenseIsValid = await checkLicense(kbnServer.server, options);
      const securityWarning = await checkSecurity(config, options);
      if (licenseIsValid === false || securityWarning === false) {
        await kbnServer.close();
        return true;
      } else {
        const count = await runner.upgrade();
        if (count > 0) {
          process.stdout.write('Performed ' + count + ' upgrade' + (count > 1 ? 's' : '') + '.\n');
        } else {
          process.stdout.write('No objects upgraded.\n');
        }
        await kbnServer.close();
        return true;
      }
    } catch (error) {
      process.stderr.write(`${error}\n`);
      await kbnServer.close();
      return false;
    }
  }

  async function restoreFromBackupFiles(config, folderPath, deleteIndicesBeforeRestoring) {
    process.stdout.write('Reverting investigate using backup files from ' + folderPath + '\n');
    const restoreKibi = new _restore_kibi2.default(config, folderPath);
    return await restoreKibi.restore(deleteIndicesBeforeRestoring);
  }

  async function processCommand(options) {
    let configFilename;
    let configPath;

    if (options.config.match(/.*investigate.yml$/) === null) {
      if (!(0, _fs.existsSync)(options.config)) {
        process.stderr.write('\nYour custom config file (' + options.config + ') was not found.\n\n' + 'Please check the --config option filepath and try again.\n\n');
        process.exit(1);
      } else {
        configFilename = (0, _path.basename)(options.config);
        configPath = options.config;
      }
    } else {
      // If there is no investigate.yml
      if (!(0, _validate_config.checkConfigYmlExists)('investigate', options.dev)) {
        process.stderr.write('\nNo config file found. ' + 'Please ensure you have an investigate.yml in the \`\/config\` folder.\n\n' + 'If you have renamed the investigate.yml, ' + 'please revert the renaming and run \`bin/investigate upgrade\` again.\n\n');
        process.exit(1);
      } else {
        // there is an investigate.yml
        configFilename = (0, _validate_config.getConfigFilename)('investigate', options.dev);
        configPath = options.dev ? (0, _from_root.fromRoot)(`config/${configFilename}`) : options.config;
      }
    }

    const config = (0, _read_yaml_config2.default)(options.config);

    if (options.dev) {
      try {
        (0, _lodash.merge)(config, (0, _read_yaml_config2.default)((0, _from_root.fromRoot)('config/investigate.dev.yml')));
      } catch (e) {
        null;
      }
    }

    const acceptAll = options.Y || false;
    const doBackup = acceptAll || !options.dontBackup;

    const folderPath = (0, _path.resolve)(options.backupDir ? options.backupDir : (0, _from_root.fromRoot)('data/'), 'backup_' + new Date().toISOString().split('.')[0].replace(/:/g, '-'));

    if (doBackup) {
      const backupKibi = new _backup_kibi2.default(config, folderPath);
      try {
        await backupKibi.backup();
      } catch (err) {
        process.stderr.write('Not able to backup indices. To skip backup step use --dont-backup flag.');
        process.exit(-1);
      }
    }

    const success = await runUpgrade(options, config);
    let deleteIndicesBeforeRestoring = false;
    if (success) {
      if (doBackup && options.deleteBackup) {
        await _rimraf2.default.sync(folderPath);
      }
      process.exit(0);
    } else {
      if (!doBackup) {
        process.exit(-1);
      } else {
        if (acceptAll) {
          process.stderr.write('There was an error during the upgrade process, automatically restoring index.\n');
          deleteIndicesBeforeRestoring = true;
          await restoreFromBackupFiles(config, folderPath, deleteIndicesBeforeRestoring);
          await _rimraf2.default.sync(folderPath);
          process.exit(-1);
        } else {
          const rlp = _readlinePromise2.default.createInterface(process.stdin, process.stdout);
          const resp = await rlp.questionAsync('There was an error and the upgrade process failed.\nDo you want to restore from backup files [N/y] ');
          const yes = resp.toLowerCase().trim()[0] === 'y';
          process.stderr.write('\n');

          const resp2 = await rlp.question('Delete index before restoring? [N/y]');
          const yes2 = resp.toLowerCase().trim()[0] === 'y';
          process.stderr.write('\n');

          if (yes) {
            if (yes2) {
              deleteIndicesBeforeRestoring = true;
            }
            await restoreFromBackupFiles(config, folderPath, deleteIndicesBeforeRestoring);
          }

          if (options.deleteBackup) {
            await _rimraf2.default.sync(folderPath);
          }

          process.exit(-1);
        }
      }
    }
  }

  program.command('upgrade').description('Upgrade saved objects').option('--dev', 'Run the upgrade using development mode configuration').option('--dont-backup', 'Run the upgrade without creating backup').option('--delete-backup', 'Delete backup files after upgrade process (overrides --keep-backup)').option('--keep-backup', 'Don\'t delete backup files after upgrade process (default)').option('-y', 'Accept all options e.g. backup the index and delete the index before restoring').option('--backup-dir <path>', 'Specify the path to the folder where the data should be saved').option('-c, --config <path>', 'Path to the config file, can be changed with the CONFIG_PATH environment variable as well', process.env.CONFIG_PATH || (0, _from_root.fromRoot)('config/investigate.yml')).option('--plugin-dir <path>', 'A path to scan for plugins, this can be specified multiple ' + 'times to specify multiple directories', pluginDirCollector, [(0, _from_root.fromRoot)('plugins'), // installed plugins
  (0, _from_root.fromRoot)('siren_plugins'), // installed siren external plugins
  (0, _from_root.fromRoot)('src/siren_core_plugins'), // siren core plugins
  (0, _from_root.fromRoot)('src/core_plugins') // kibana plugins
  ]).action(processCommand);
};

var _kbn_server = require('../../server/kbn_server');

var _kbn_server2 = _interopRequireDefault(_kbn_server);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _rimraf = require('rimraf');

var _rimraf2 = _interopRequireDefault(_rimraf);

var _commonTags = require('common-tags');

var _lodash = require('lodash');

var _validate_config = require('../../cli/kibi/validate_config');

var _read_yaml_config = require('../serve/read_yaml_config');

var _read_yaml_config2 = _interopRequireDefault(_read_yaml_config);

var _path = require('path');

var _from_root = require('../../utils/from_root');

var _migration_runner = require('kibiutils/lib/migrations/migration_runner');

var _migration_runner2 = _interopRequireDefault(_migration_runner);

var _migration_logger = require('kibiutils/lib/migrations/migration_logger');

var _migration_logger2 = _interopRequireDefault(_migration_logger);

var _readlinePromise = require('readline-promise');

var _readlinePromise2 = _interopRequireDefault(_readlinePromise);

var _license_format_check = require('./license_format_check');

var _license_format_check2 = _interopRequireDefault(_license_format_check);

var _fs = require('fs');

var _backup_kibi = require('./_backup_kibi');

var _backup_kibi2 = _interopRequireDefault(_backup_kibi);

var _restore_kibi = require('./_restore_kibi');

var _restore_kibi2 = _interopRequireDefault(_restore_kibi);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const pathCollector = function () {
  const paths = [];
  return function (path) {
    paths.push((0, _path.resolve)(process.cwd(), path));
    return paths;
  };
};

const pluginDirCollector = pathCollector();

/**
 * The command to upgrade saved objects.
 */
;
module.exports = exports['default'];
