'use strict';

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _fs = require('fs');

var _cluster = require('cluster');

var _path = require('path');

var _utils = require('../../utils');

var _path2 = require('../../server/path');

var _read_yaml_config = require('./read_yaml_config');

var _read_yaml_config2 = _interopRequireDefault(_read_yaml_config);

var _dev_ssl = require('../dev_ssl');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let canCluster;
try {
  require.resolve('../cluster/cluster_manager');
  canCluster = true;
} catch (e) {
  canCluster = false;
}

const pathCollector = function () {
  const paths = [];
  return function (path) {
    paths.push((0, _path.resolve)(process.cwd(), path));
    return paths;
  };
};

//kibi: Check if there is a plugins folder and return its path if it does
const getPluginsFolder = function (path) {
  try {
    (0, _fs.statSync)((0, _utils.fromRoot)(path));
    return (0, _utils.fromRoot)(path);
  } catch (err) {
    console.warn('[WARNING] ' + path + ' folder is missing, please create one and restart Siren Investigate ...');
    return null;
  }
};
// kibi

const configPathCollector = pathCollector();
const pluginDirCollector = pathCollector();
const pluginPathCollector = pathCollector();

function readServerSettings(opts, extraCliOptions) {
  const settings = (0, _read_yaml_config2.default)(opts.config);
  const set = _lodash2.default.partial(_lodash2.default.set, settings);
  const get = _lodash2.default.partial(_lodash2.default.get, settings);
  const has = _lodash2.default.partial(_lodash2.default.has, settings);
  const merge = _lodash2.default.partial(_lodash2.default.merge, settings);

  set('configFiles', opts.config);

  if (opts.dev) {
    set('env', 'development');
    set('optimize.lazy', true);

    if (opts.ssl) {
      set('server.ssl.enabled', true);
    }

    if (opts.ssl && !has('server.ssl.certificate') && !has('server.ssl.key')) {
      set('server.ssl.certificate', _dev_ssl.DEV_SSL_CERT_PATH);
      set('server.ssl.key', _dev_ssl.DEV_SSL_KEY_PATH);
    }
  }

  if (opts.elasticsearch) set('elasticsearch.url', opts.elasticsearch);
  if (opts.port) set('server.port', opts.port);
  if (opts.host) set('server.host', opts.host);
  if (opts.quiet) set('logging.quiet', true);
  if (opts.silent) set('logging.silent', true);
  if (opts.verbose) set('logging.verbose', true);
  if (opts.logFile) set('logging.dest', opts.logFile);

  // kibi: let's disable selected plugins. This is done for ui testing
  if (opts.sirenDisablePlugins) {
    set('siren.disabledPlugins', ['gremlin_server', 'siren_federate']);
  };
  // kibi: end

  // kibi: Stop these plugins being disabled (for testing)
  if (opts.forcePlugin) {
    set('siren.forcedPlugins', _lodash2.default.compact([].concat(get('siren.forcedPlugins'), opts.forcePlugin)));
  }
  // kibi

  set('plugins.scanDirs', _lodash2.default.compact([].concat(get('plugins.scanDirs'), opts.pluginDir)));

  set('plugins.paths', _lodash2.default.compact([].concat(get('plugins.paths'), opts.pluginPath)));

  merge(extraCliOptions);

  return settings;
}

module.exports = function (program) {
  const command = program.command('serve');

  command.description('Run the kibana server').collectUnknownOptions().option('-e, --elasticsearch <uri>', 'Elasticsearch instance').option('-c, --config <path>', 'Path to the config file, can be changed with the CONFIG_PATH environment variable as well. ' + 'Use multiple --config args to include multiple config files.', configPathCollector, [(0, _path2.getConfig)()]).option('-p, --port <port>', 'The port to bind to', parseInt).option('-q, --quiet', 'Prevent all logging except errors').option('-Q, --silent', 'Prevent all logging').option('--verbose', 'Turns on verbose logging').option('-H, --host <host>', 'The host to bind to').option('-l, --log-file <path>', 'The file to log to')
  // kibi: adds option to quit server after optimization
  .option('--quit-after-init', 'Quit the server after a successful initialization.')
  // kibi: end
  .option('--plugin-dir <path>', 'A path to scan for plugins, this can be specified multiple ' + 'times to specify multiple directories', pluginDirCollector, [getPluginsFolder('plugins'), // kibi: Check if plugins folder exists
  getPluginsFolder('siren_plugins'), // kibi: Check if siren_plugins folder exists
  (0, _utils.fromRoot)('src/siren_core_plugins'), // kibi: point to siren core plugins
  (0, _utils.fromRoot)('src/core_plugins')]).option('--plugin-path <path>', 'A path to a plugin which should be included by the server, ' + 'this can be specified multiple times to specify multiple paths', pluginPathCollector, []).option('--plugins <path>', 'an alias for --plugin-dir', pluginDirCollector);

  if (canCluster) {
    command.option('--dev', 'Run the server with development mode defaults').option('--no-ssl', 'Don\'t run the dev server using HTTPS').option('--no-base-path', 'Don\'t put a proxy in front of the dev server, which adds a random basePath').option('--no-watch', 'Prevents automatic restarts of the server in --dev mode')
    // kibi: extra option added to be able to ignore investigate.dev.yml when running tests
    .option('--ignoreDevYml', 'Allow to ignore investigate.dev.yml file when running in dev mode. ' + 'Useful for running tests like: grunt test:dev --kbnServer.ignoreDevYml').option('--forcePlugin <plugin>', 'Ignore disabling of the specified plugin (for testing purposes)', []).option('--sirenDisablePlugins', 'Tell Investigate not to look for some plugins so we can run UI tests without them');
  }

  command.action(async function (opts) {
    if (opts.dev && !opts.ignoreDevYml) {
      try {
        const kbnDevConfig = (0, _utils.fromRoot)('config/investigate.dev.yml'); // kibi: renamed kibana to investigate
        if ((0, _fs.statSync)(kbnDevConfig).isFile()) {
          opts.config.push(kbnDevConfig);
        }
      } catch (err) {
        // ignore, kibana.dev.yml does not exist
      }
    }

    // kibi: added extra condition !opts.ignoreDevYml

    const getCurrentSettings = () => readServerSettings(opts, this.getUnknownOptions());
    const settings = getCurrentSettings();

    if (canCluster && opts.dev && !_cluster.isWorker) {
      // stop processing the action and handoff to cluster manager
      const ClusterManager = require('../cluster/cluster_manager');
      new ClusterManager(opts, settings);
      return;
    }

    let kbnServer = {};
    const KbnServer = require('../../server/kbn_server');
    try {
      kbnServer = new KbnServer(settings);
      await kbnServer.ready();
      // kibi: adds option to quit server after optimization
      if (opts.quitAfterInit) {
        process.exit(0);
      }
      // kibi: end
    } catch (err) {
      const { server } = kbnServer;

      if (err.code === 'EADDRINUSE') {
        logFatal(`Port ${err.port} is already in use. Another instance of Investigate may be running!`, server);
      } else {
        logFatal(err, server);
      }

      kbnServer.close();
      process.exit(1); // eslint-disable-line no-process-exit
    }

    process.on('SIGHUP', function reloadConfig() {
      const settings = getCurrentSettings();
      kbnServer.server.log(['info', 'config'], 'Reloading logging configuration due to SIGHUP.');
      kbnServer.applyLoggingConfiguration(settings);
      kbnServer.server.log(['info', 'config'], 'Reloaded logging configuration due to SIGHUP.');
    });

    return kbnServer;
  });
};

function logFatal(message, server) {
  if (server) {
    server.log(['fatal'], message);
  }
  console.error('FATAL', message);
}
