import angular from 'angular';
import _ from 'lodash';
import { uiModules } from 'ui/modules';
// kibi: dependecies
import uuid from 'uuid';
import { findMainCoatNodeSearchId } from 'ui/kibi/components/dashboards360/coat_tree';
import { JoinAlgorithmType } from 'ui/kibi/components/dashboards360/lib/join_algorithm_type';
import { DashboardDatamodelType } from './dashboard_datamodel_type';
// kibi: end

const module = uiModules.get('app/dashboard');

// Used only by the savedDashboards service, usually no reason to change this
module.factory('SavedDashboard', function (courier, config, $rootScope, savedSearches, indexPatterns) {
  // SavedDashboard constructor. Usually you'd interact with an instance of this.
  // ID is option, without it one will be generated on save.
  _.class(SavedDashboard).inherits(courier.SavedObject);
  function SavedDashboard(id) {
    // Gives our SavedDashboard the properties of a SavedObject
    SavedDashboard.Super.call(this, {
      type: SavedDashboard.type,
      mapping: SavedDashboard.mapping,
      searchSource: SavedDashboard.searchsource,

      // if this is null/undefined then the SavedObject will be assigned the defaults
      id: id,

      // default values that will get assigned if the doc is new
      defaults: {
        title: 'New Dashboard',
        hits: 0,
        description: '',
        panelsJSON: '[]',
        optionsJSON: angular.toJson({
          darkTheme: config.get('dashboard:defaultDarkTheme'),
          hideBorders: false, //kibi: set it false as default,
          hideSearchBar: false,
          showTimePicker: false //kibi: set it false as default,
        }),
        uiStateJSON: '{}',
        version: 2, // changed to 2 after adding coatJSON
        timeRestore: false,
        timeMode: undefined, // kibi: save the mode of the timepicker
        timeTo: undefined,
        timeFrom: undefined,
        refreshInterval: undefined,
        priority: 0, // kibi: added to sort dashboards
        coatJSON: '{}' // kibi: added to hold the tree of center of aggregation for dashboards 360
      },

      // if an indexPattern was saved with the searchsource of a SavedDashboard
      // object, clear it. It was a mistake
      clearSavedIndexPattern: true
    });

    // kibi: extends save function to notify the end of the save operation via event.
    const save = this.save;
    this.save = options => {
      return save(options).then(result => {
        $rootScope.$broadcast('siren:dashboard:saved', this.id);
        return result;
      });
    };
    // kibi: end
  }

  // save these objects with the 'dashboard' type
  SavedDashboard.type = 'dashboard';

  // if type:dashboard has no mapping, we push this mapping into ES
  SavedDashboard.mapping = {
    title: 'string',
    hits: 'integer',
    description: 'string',
    panelsJSON: 'string',
    optionsJSON: 'string',
    uiStateJSON: 'string',
    version: 'integer',
    timeRestore: 'boolean',
    // kibi: 'timeMode' is added to mapping
    timeMode: 'string',
    timeTo: 'string',
    timeFrom: 'string',
    refreshInterval: {
      type: 'object',
      properties: {
        display: { type: 'string' },
        pause: { type: 'boolean' },
        section: { type: 'integer' },
        value: { type: 'integer' }
      }
    },
    priority: 'integer', // kibi: added to sort dashboards
    coatJSON: 'string' // kibi: 'coatJSON' is added to mapping
  };

  // Order these fields to the top, the rest are alphabetical
  SavedDashboard.fieldOrder = ['title', 'description'];

  SavedDashboard.searchsource = true;

  function shouldSetUseGlobalTimeFilter(searchId, searchesList, patternsList) {
    if (!searchesList || !patternsList) {
      return false;
    }

    const search = _.find(searchesList, 'id', searchId);
    if (!search) {
      return false;
    }
    const searchSource = JSON.parse(search.kibanaSavedObjectMeta.searchSourceJSON);
    const indexPattern = _.find(patternsList, 'id', searchSource.index);
    if (!indexPattern) {
      return false;
    }

    return !!indexPattern.timeFieldName;
  }

  function getCoatWithJustMainNode(mainSavedSearchId, searches, indexPatterns) {
    const coat = JSON.parse(this.coatJSON);
    coat.items = [{
      type: 'node',
      id: uuid.v1(),
      d: {
        entity: {
          id: mainSavedSearchId
        },
        isRoot: true,
        widgets: []
      }
    }];

    if (shouldSetUseGlobalTimeFilter(mainSavedSearchId, searches, indexPatterns)) {
      coat.items[0].d.entity.useGlobalTimeFilter = true;
    }

    coat.datamodelType = DashboardDatamodelType.SINGLE_SEARCH;
    return JSON.stringify(coat);
  }

  SavedDashboard.prototype.getJoinAlgorithmType = function () {
    const parsedCoatJson = JSON.parse(this.coatJSON);
    return parsedCoatJson.joinAlgorithmType;
  };

  SavedDashboard.prototype.setJoinAlgorithmType = function (joinAlgorithmType) {
    const parsedCoatJson = JSON.parse(this.coatJSON);
    if (parsedCoatJson.datamodelType === DashboardDatamodelType.MULTIPLE_SEARCHES) {
      parsedCoatJson.joinAlgorithmType = joinAlgorithmType;
    } else {
      delete parsedCoatJson.joinAlgorithmType;
    }

    this.coatJSON = JSON.stringify(parsedCoatJson);
  };

  SavedDashboard.prototype.getDashboardDataModelType = function () {
    const parsedCoatJson = JSON.parse(this.coatJSON);
    return parsedCoatJson.datamodelType || DashboardDatamodelType.NO_SEARCH;
  };

  SavedDashboard.prototype.setDashboardDataModelType = function (datamodelType) {
    const parsedCoatJson = JSON.parse(this.coatJSON);
    // we have to be able to distinguish between
    // sinle search dash AND multiple search dash (technically the same but user can mark a dash as one or another)
    if (
      datamodelType === DashboardDatamodelType.NO_SEARCH ||
      datamodelType === DashboardDatamodelType.SINGLE_SEARCH
    ) {
      parsedCoatJson.datamodelType = datamodelType;
      parsedCoatJson.items = [];
      delete parsedCoatJson.viewSettings;
      delete parsedCoatJson.joinAlgorithmType;
    } else if (datamodelType === DashboardDatamodelType.MULTIPLE_SEARCHES) {
      parsedCoatJson.datamodelType = datamodelType;
    } else {
      throw new Error('Unsupported DashboardDatamodelType');
    }

    this.coatJSON = JSON.stringify(parsedCoatJson);
  };

  SavedDashboard.prototype.getMainSavedSearchId = function () {
    // we cache this property to avoid recomputing it everytime (as it is used quite often)
    if (this.$$cachedMainSavedSearchId) {
      return this.$$cachedMainSavedSearchId;
    }
    const mainSavedSearchId = findMainCoatNodeSearchId(this.coatJSON);
    if (mainSavedSearchId) {
      this.$$cachedMainSavedSearchId = mainSavedSearchId;
    }
    return mainSavedSearchId;
  };

  SavedDashboard.prototype.setMainSavedSearchId = function (mainSavedSearchId) {
    const patternsListPromise = indexPatterns.find()
      .then(resp => {
        return resp.savedObjects.map(savedObject => {
          return {
            id: savedObject.id,
            ... savedObject.attributes
          };
        });
      });
    const searchesListPromise = savedSearches.find().then(resp => resp.hits);

    return Promise.all([searchesListPromise, patternsListPromise])
      .then(res => {
        const searchesList = res[0];
        const patternsList = res[1];
        this.coatJSON = getCoatWithJustMainNode.call(this, mainSavedSearchId, searchesList, patternsList);
        this.$$cachedMainSavedSearchId = mainSavedSearchId; // must be after this.coatJSON as the former is resetting $$cachedMainSavedSearchId
        $rootScope.$emit('savedDashboard:setMainSearchId', this.id, mainSavedSearchId);
      });
  };

  const emptyCoatJSON = JSON.stringify({
    joinAlgorithmType: JoinAlgorithmType.INNER_FILTER_JOIN,
    datamodelType: DashboardDatamodelType.NO_SEARCH
  });

  Object.defineProperty(SavedDashboard.prototype, 'coatJSON', {
    enumerable: true,
    get() {
      return this.$$cachedCoatJSON || emptyCoatJSON;
    },
    set(coatJSON) {
      delete this.$$cachedMainSavedSearchId; // always clear cached main search id when setting new coatJSON
      this.$$cachedCoatJSON = coatJSON;
    }
  });

  // kibi: define a getter and setter for backward compability where old savedSearchId property was used
  Object.defineProperty(SavedDashboard.prototype, 'savedSearchId', {
    get() {
      console.log('WARNING: old savedSearchId property was used instead of getMainSavedSearchId'); // eslint-disable-line no-console
      return this.getMainSavedSearchId();
    },
    set(mainSavedSearchId) {
      console.log('WARNING: old savedSearchId property was used instead of setMainSavedSearchId'); // eslint-disable-line no-console
      this.coatJSON = getCoatWithJustMainNode.call(this, mainSavedSearchId);
    }
  });

  return SavedDashboard;
});
