'use strict';

var _child_process = require('child_process');

var _child_process2 = _interopRequireDefault(_child_process);

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _requestPromise = require('request-promise');

var _requestPromise2 = _interopRequireDefault(_requestPromise);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _os = require('os');

var _os2 = _interopRequireDefault(_os);

var _url = require('url');

var _url2 = _interopRequireDefault(_url);

var _gremlin_server_errors = require('./gremlin_server_errors');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let startInterval;

function getRejecthandler(self, reject) {
  return function (message) {
    let error = new _gremlin_server_errors.GremlinError(message);
    if (message instanceof Error) {
      error = message;
    }
    clearInterval(startInterval);
    self.initializing = false;
    reject(error);
  };
}

function getFulfillHandler(self, fulfill) {
  return function (message) {
    clearInterval(startInterval);
    self.initialized = true;
    self.initializing = false;
    fulfill({ message });
  };
}

function startServer(self, fulfill, reject, gremlinStartupTimeout) {
  const fulfillHandler = getFulfillHandler(self, fulfill);
  const rejectHandler = getRejecthandler(self, reject);
  // NOTE:
  // the whole start operation can NOT take longer than gremlinStartTimeout
  // this is to prevent the situation where start takes longer than mocha test timeout
  // and mocha kills the test leaving the gremlin server process hanging
  const startTime = new Date().getTime();
  startInterval = setInterval(function () {
    const now = new Date().getTime();
    const elapsed = now - startTime;
    if (elapsed >= self.gremlinStartTimeout) {
      // here we have to clear immediatelly
      clearInterval(startInterval);
      self.stop('handler - could not properly start before timeout: ' + self.gremlinStartTimeout).then(() => {
        rejectHandler('The Siren Gremlin Server did not start correctly before the gremlinStartTimeout');
      });
    }
  }, 1000);

  if (self.initializing === true) {
    // exit as initalization is already in progress
    const msg = 'Initialization of Siren Gremlin Server in progress. Won\'t start another one.';
    self.server.log(['gremlin', 'warning'], msg);
    return;
  }

  self.initializing = true;
  const config = self.server.config();
  self.url = config.get('investigate_core.gremlin_server.url');
  self._isAnotherGremlinRunning().then(() => {
    const message = 'Another Siren Gremlin Server was found running. Won\'t start another instance.';
    self.server.log(['gremlin', 'warning'], message);
    fulfillHandler(message);
  }).catch(() => {
    if (config.has('investigate_core.gremlin_server.path')) {
      let gremlinServerPath = config.get('investigate_core.gremlin_server.path');

      isJavaVersionOk(self).then(function () {

        if (config.has('investigate_core.gremlin_server.ssl.ca')) {
          const ca = config.get('investigate_core.gremlin_server.ssl.ca');
          if (ca) {
            try {
              self.ca = _fs2.default.readFileSync(ca);
            } catch (error) {
              rejectHandler('The configuration property investigate_core.gremlin_server.ca does not point to a readable CA file.');
            }
          }
        }

        if (_path2.default.parse(gremlinServerPath).ext !== '.jar') {
          rejectHandler(`The configuration property investigate_core.gremlin_server.path [${gremlinServerPath}] does not point to a jar file`);
        }

        if (!_path2.default.isAbsolute(gremlinServerPath)) {
          const rootDir = _path2.default.normalize(__dirname + _path2.default.sep + '..' + _path2.default.sep + '..' + _path2.default.sep + '..' + _path2.default.sep);
          const gremlinDirtyDir = _path2.default.join(rootDir, gremlinServerPath);
          gremlinServerPath = _path2.default.resolve(_path2.default.normalize(gremlinDirtyDir));
        }

        return _fs2.default.access(gremlinServerPath, _fs2.default.F_OK, error => {
          if (error !== null) {
            rejectHandler(`The Siren Gremlin Server jar file was not found at [${gremlinServerPath}]. Please check the configuration`);
          } else {
            const serverURL = _url2.default.parse(self.url);
            const esUrl = config.get('elasticsearch.url');

            const args = ['-jar', gremlinServerPath, '-Djava.security.egd=file:/dev/./urandom', '--elasticsearch.url=' + esUrl, '--server.port=' + serverURL.port];
            if (serverURL.hostname !== '0.0.0.0') {
              args.push('--server.address=' + serverURL.hostname);
            }

            if (config.has('kibana.index')) {
              const kibanaIndex = config.get('kibana.index');
              if (kibanaIndex) {
                args.push('--kibi.index=' + kibanaIndex);
              }
            }

            try {
              if (config.has('configFiles')) {
                const configFiles = config.get('configFiles');
                if (configFiles) {
                  for (const configFile of configFiles) {
                    args.push(`--yaml=${configFile}`);
                  }
                }
              }
            } catch (e) {
              rejectHandler('The Siren Gremlin Server cannot be started as the reference to the yml ' + 'configuration was not set.');
            }

            if (config.has('investigate_core.gremlin_server.debug_remote')) {
              const gremlinServerRemoteDebug = config.get('investigate_core.gremlin_server.debug_remote');
              if (gremlinServerRemoteDebug) {
                args.unshift(gremlinServerRemoteDebug);
              }
            }

            if (config.has('investigate_core.gremlin_server.log_conf_path')) {
              const logConfigPath = config.get('investigate_core.gremlin_server.log_conf_path');
              if (logConfigPath) {
                args.push('--logging.config=' + logConfigPath);
              }
            }

            if (config.has('elasticsearch.ssl.verificationMode')) {
              const verificationMode = config.get('elasticsearch.ssl.verificationMode');
              if (['none', 'full', 'certificate'].indexOf(verificationMode) === -1) {
                const message = `Unknown ssl verificationMode: ${verificationMode} ` + 'while starting the Siren Gremlin Server';
                self.initializing = false;
                rejectHandler(message);
              }
            }

            if (config.has('investigate_core.gremlin_server.ssl.key_store') && config.get('investigate_core.gremlin_server.ssl.key_store')) {
              const sslKeyStorePsw = config.get('investigate_core.gremlin_server.ssl.key_store_password');
              if (!sslKeyStorePsw) {
                rejectHandler('The Siren Gremlin Server keystore password was not specified; ' + 'in investigate_core.gremlin_server.ssl.key_store_password');
              }
            } else if (config.has('investigate_access_control.enabled') && config.get('investigate_access_control.enabled') && !self._isLocalhost(serverURL.hostname)) {
              rejectHandler('Since you are using access control, you must enable HTTPS support in Siren Gremlin Server ' + 'by configuring the key store in investigate.yml\n' + 'The following properties are required:\n' + 'investigate_core.gremlin_server.ssl.key_store\n' + 'investigate_core.gremlin_server.ssl.key_store_password\n' + 'investigate_core.gremlin_server.ssl.ca (optional)');
            }

            const quietMode = config.has('investigate_core.gremlin_server.quiet') ? config.get('investigate_core.gremlin_server.quiet') : false;

            self.server.log(['gremlin', 'info'], 'Starting the Siren Gremlin Server');
            self.gremlinServer = _child_process2.default.spawn('java', args);

            if (!quietMode) {
              self.gremlinServer.stdout.on('data', data => {
                const stringData = ('' + data).trim();
                // Quick test to skip all strings that do not start with 'WARN/WARNING'
                if (stringData.indexOf('W') === 0 || stringData.indexOf('The parameter include_type_name should be explicitly specified') !== -1) {
                  // The warning emitted when running 6.6 < ES version < 7
                } else {
                  return self.server.log(['gremlin stdout stream', 'info'], stringData);
                }
              });
              self.gremlinServer.stderr.on('data', data => {
                const stringData = ('' + data).trim();
                // Quick test to skip all strings that do not start with 'WARNING'
                if (stringData.indexOf('W') === 0 && stringData.indexOf('WARNING: Illegal reflective access by org.codehaus.groovy') !== -1) {
                  // Hide the warning emitted by groovy with JAVA >= 9
                } else {
                  return self.server.log(['gremlin stderr stream', 'error'], stringData);
                }
              });

              self.gremlinServer.on('error', err => {
                // eslint-disable-line memoryleaks
                rejectHandler(err.message);
              });
            }

            const maxCounter = 20;
            const initialTimeout = gremlinStartupTimeout > 15000 ? 15000 : 0;
            const timeout = (gremlinStartupTimeout - initialTimeout) / maxCounter;
            const counter = maxCounter;

            self.ping = function (counter) {
              if (counter > 0) {
                setTimeout(function () {
                  self._ping().then(function (resp) {
                    let jsonResp;
                    try {
                      jsonResp = JSON.parse(resp.toString());
                    } catch (err) {
                      jsonResp = resp;
                    }
                    if (jsonResp.status === 'ok') {
                      const message = 'The Siren gremlin server started successfully at ' + self.url;
                      self.server.log(['gremlin', 'info'], message);
                      fulfillHandler(message);
                    } else {
                      self.server.log(['gremlin', 'warning'], 'Waiting for the Siren Gremlin Server. Ping response was ' + jsonResp);
                      counter--;
                      setTimeout(() => self.ping(counter), timeout);
                    }
                  }).catch(function (err) {
                    if (err.error && err.error.code !== 'ECONNREFUSED') {
                      self.server.log(['gremlin', 'error'], 'Failed to ping the Siren Gremlin Server: ' + err.message);
                    } else {
                      self.server.log(['gremlin', 'warning'], 'Waiting for the Siren Gremlin Server.');
                      self.server.log(['gremlin', 'debug'], 'Error ' + JSON.stringify(err));
                    }
                    counter--;
                    setTimeout(() => self.ping(counter), timeout);
                  });
                }, counter === maxCounter ? initialTimeout : timeout);
              } else {
                const message = `The Siren Gremlin Server did not start correctly after ${maxCounter} pings`;
                self.stop(`handler - ${message}`).then(() => {
                  rejectHandler(message);
                });
              }
            };
            self.ping(counter);
          }
        });
      }).catch(err => {
        rejectHandler(err);
      });
    } else {
      rejectHandler(`The Siren Gremlin Server jar file was not found. Please check the
        value of the investigate_core.gremlin_server.path configuration property.`);
    }
  });
};

async function isJavaVersionOk(self) {
  return new _bluebird2.default((fulfill, reject) => {
    const spawn = require('child_process').spawn('java', ['-version']);
    spawn.on('error', function (err) {
      if (err.code === 'ENOENT') {
        self.server.log(['gremlin', 'error'], 'Java not found, please ensure that ' + 'JAVA_HOME is set correctly and the Java binaries are in the application path');
        self.javaCheck = { checked: true, isOk: false };
        reject(new _gremlin_server_errors.GremlinError('Java not found'));
      }
    });
    spawn.stderr.on('data', function (data) {
      const err = self._checkJavaVersionString(data);
      if (err) {
        reject(new _gremlin_server_errors.GremlinError(err));
      } else if (self.javaCheck.isOk) {
        fulfill(true);
      }
    });
  });
}

const javaNumberRegex = /(\d+?)\.(\d+?)\.(\d+?)(?:_(\d+))?/;

class GremlinServerHandler {
  constructor(server, gremlinStartTimeout) {
    this.gremlinServer = null;
    this.initializing = false;
    this.initialized = false;
    this.server = server;
    this.javaCheck = { checked: false, isOk: null };
    this.gremlinStartTimeout = gremlinStartTimeout || 180000;
  }

  _isLocalhost(URLHostname) {
    return URLHostname === '127.0.0.1' || URLHostname === '::1' || URLHostname === 'localhost';
  }

  async _isAnotherGremlinRunning() {
    return new _bluebird2.default((fulfill, reject) => {
      this._ping().then(function (resp) {
        const jsonResp = JSON.parse(resp.toString());
        if (jsonResp.status === 'ok') {
          fulfill();
        } else {
          reject(new _gremlin_server_errors.GremlinError('GremlinServer is not running'));
        }
      }).catch(err => {
        reject(new _gremlin_server_errors.GremlinError(err.message));
      });
    });
  }

  _checkJavaVersionString(string) {
    if (!this.javaCheck.checked || this.javaCheck.checked && !this.javaCheck.isOk) {
      let err;
      const versionLine = _lodash2.default.find(string.toString().split(_os2.default.EOL), line => {
        if (line.indexOf(' version ') !== -1) {
          const numbers = line.match(javaNumberRegex);
          if (numbers && numbers.length >= 2) {
            return true;
          }
        }
        return false;
      });
      if (versionLine && versionLine.match(javaNumberRegex)) {
        //[string, major, minor, patch, update, ...]
        const matches = versionLine.match(javaNumberRegex);
        this.javaCheck.isOk = false;
        if (matches.length >= 2) {
          // Check for JAVA 8
          if (matches[1] === '1' && this._isJavaVersionCompatible(matches[2])) {
            this.javaCheck.isOk = true;
            // Check for JAVA >= 9
          } else if (matches[1] !== '1' && this._isJavaVersionCompatible(matches[1])) {
            this.javaCheck.isOk = true;
          }
        }
        if (!this.javaCheck.isOk) {
          err = 'Java version is lower than the requested 1.8. The Siren Gremlin Server needs Java 8 to run';
        }
      } else {
        this.javaCheck.isOk = false;
      }
      this.javaCheck.checked = true;
      return err;
    } else {
      return null;
    }
  }

  /**
   * Checks that the passed string is a number and is >= 8,
   * which is the minimum requirement for the gremlin server.
   */
  _isJavaVersionCompatible(string) {
    // Check if the string is a number
    if (!isNaN(string)) {
      const num = parseInt(string);
      if (num >= 8) {
        return true;
      } else {
        return false;
      }
    } else {
      return false;
    }
  }

  _isProcessRunning(pid) {
    try {
      process.kill(pid, 0);
      return true;
    } catch (e) {
      return e.code === 'EPERM'; // EPERM indicate the permission issues
    }
  }

  async _waitUntilProcessDie(pid) {
    const waitTimeinMs = 250;
    const running = this._isProcessRunning(pid);
    if (!running) {
      this.server.log(['gremlin', 'info'], `Siren Gremlin Server process ${pid} no longer exist`);
      return _bluebird2.default.resolve(true);
    }
    this.server.log(['gremlin', 'info'], `Waiting ${waitTimeinMs}ms for Siren Gremlin Server process ${pid} to die`);
    return new _bluebird2.default((fulfill, reject) => {
      setTimeout(() => {
        fulfill(this._waitUntilProcessDie(pid));
      }, waitTimeinMs);
    });
  }

  async start(who = 'UNKNOWN') {
    this.server.log(['gremlin', 'info'], 'Starting the Siren Gremlin Server ' + who);

    if (this.initialized) {
      return _bluebird2.default.resolve({
        message: 'GremlinServerHandler already initialized'
      });
    }

    return new _bluebird2.default((fulfill, reject) => {
      startServer(this, fulfill, reject, this.gremlinStartTimeout);
    });
  }

  async stop(who = 'UNKNOWN') {

    if (this.initialized) {
      this.initialized = false;
      return new _bluebird2.default((fulfill, reject) => {
        this.server.log(['gremlin', 'info'], 'Stopping the Siren Gremlin Server ' + who);

        if (this.gremlinServer) {
          const pid = this.gremlinServer.pid;
          const exitCode = this.gremlinServer.kill('SIGINT');
          if (exitCode) {
            // check if the process is running
            this._waitUntilProcessDie(pid).then(() => {
              this.server.log(['gremlin', 'info'], 'The Siren Gremlin Server exited successfully');
              fulfill(true);
            });
          } else {
            this.server.log(['gremlin', 'error'], 'The Siren Gremlin Server did not accepted the SIGINT, status: ' + exitCode);
            reject(new _gremlin_server_errors.GremlinStopError('The Siren Gremlin Server did not accepted the SIGINT, status: ' + exitCode));
          }
        } else {
          fulfill(true);
        }
      });
    } else {
      // If the server did not start no need to kill it
      return _bluebird2.default.resolve(true);
    }
  }

  isInitialized() {
    return this.initialized;
  }

  _ping() {
    const options = {
      method: 'GET',
      uri: this.url + '/ping'
    };
    if (this.ca) {
      options.ca = this.ca;
    }
    return (0, _requestPromise2.default)(options);
  }

  /*
   * Used in tests.
   */
  _getJavaCheck() {
    return this.javaCheck;
  }
}

module.exports = GremlinServerHandler;
