'use strict';

var _lodash = require('lodash');

var _bluebird = require('bluebird');

var _cluster = require('cluster');

var _utils = require('../utils');

var _config = require('./config/config');

var _config2 = _interopRequireDefault(_config);

var _configuration = require('./logging/configuration');

var _configuration2 = _interopRequireDefault(_configuration);

var _setup = require('./config/setup');

var _setup2 = _interopRequireDefault(_setup);

var _http = require('./http');

var _http2 = _interopRequireDefault(_http);

var _logging = require('./logging');

var _logging2 = _interopRequireDefault(_logging);

var _warnings = require('./warnings');

var _warnings2 = _interopRequireDefault(_warnings);

var _status = require('./status');

var _status2 = _interopRequireDefault(_status);

var _pid = require('./pid');

var _pid2 = _interopRequireDefault(_pid);

var _scan = require('./plugins/scan');

var _scan2 = _interopRequireDefault(_scan);

var _check_enabled = require('./plugins/check_enabled');

var _check_enabled2 = _interopRequireDefault(_check_enabled);

var _check_version = require('./plugins/check_version');

var _check_version2 = _interopRequireDefault(_check_version);

var _complete = require('./config/complete');

var _complete2 = _interopRequireDefault(_complete);

var _deprecation_warnings = require('./config/deprecation_warnings');

var _ui = require('../ui');

var _ui2 = _interopRequireDefault(_ui);

var _optimize = require('../optimize');

var _optimize2 = _interopRequireDefault(_optimize);

var _initialize = require('./plugins/initialize');

var _initialize2 = _interopRequireDefault(_initialize);

var _index_patterns = require('./index_patterns');

var _saved_objects = require('./saved_objects');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const rootDir = (0, _utils.fromRoot)('.');

module.exports = class KbnServer {
  constructor(settings) {
    this.name = _utils.pkg.name;
    this.version = _utils.pkg.version;
    this.branch = _utils.pkg.branch;
    this.kibiVersion = _utils.pkg.kibi_version; // kibi: added to manage kibi version
    this.kibiKibanaAnnouncement = _utils.pkg.kibi_kibana_announcement; // kibi: added by kibi
    this.build = _utils.pkg.build || false;
    this.rootDir = rootDir;
    this.settings = settings || {};
    this.cleaningArray = []; // Kibi: added to manage cleanup functions through shutdown_manager
    this.migrations = {}; // kibi: used to register migrations in preInit method

    this.ready = (0, _lodash.constant)(this.mixin(
    // sets this.config, reads this.settings
    _setup2.default,
    // sets this.server
    _http2.default, _logging2.default, _deprecation_warnings.configDeprecationWarningsMixin, _warnings2.default, _status2.default,

    // writes pid file
    _pid2.default,

    // find plugins and set this.plugins
    _scan2.default,

    // disable the plugins that are disabled through configuration
    _check_enabled2.default,

    // disable the plugins that are incompatible with the current version of Kibana
    _check_version2.default,

    // tell the config we are done loading plugins
    _complete2.default,

    // setup this.uiExports and this.bundles
    _ui2.default, _index_patterns.indexPatternsMixin,

    // setup saved object routes
    _saved_objects.savedObjectsMixin,

    // ensure that all bundles are built, or that the
    // lazy bundle server is running
    _optimize2.default,

    // finally, initialize the plugins
    _initialize2.default, () => {
      if (this.config.get('server.autoListen')) {
        this.ready = (0, _lodash.constant)((0, _bluebird.resolve)());
        return this.listen();
      }
    }));

    this.listen = (0, _lodash.once)(this.listen);
  }

  /**
   * Extend the KbnServer outside of the constraits of a plugin. This allows access
   * to APIs that are not exposed (intentionally) to the plugins and should only
   * be used when the code will be kept up to date with Kibana.
   *
   * @param {...function} - functions that should be called to mixin functionality.
   *                         They are called with the arguments (kibana, server, config)
   *                         and can return a promise to delay execution of the next mixin
   * @return {Promise} - promise that is resolved when the final mixin completes.
   */
  async mixin(...fns) {
    for (const fn of (0, _lodash.compact)((0, _lodash.flatten)(fns))) {
      await fn.call(this, this, this.server, this.config);
    }
  }

  /**
   * Tell the server to listen for incoming requests, or get
   * a promise that will be resolved once the server is listening.
   *
   * @return undefined
   */
  async listen() {
    // kibi: config is added here for 'shutdown_manager'
    const { server, config } = this;

    await this.ready();
    await (0, _bluebird.fromNode)(cb => server.start(cb));
    await require('./shutdown_manager')(this, server); // kibi: added here to manage pid and gremlin server shutdown

    if (_cluster.isWorker) {
      // help parent process know when we are ready
      process.send(['WORKER_LISTENING']);
    }

    server.log(['listening', 'info'], `Server running at ${server.info.uri}`);
    return server;
  }

  async close() {
    await (0, _bluebird.fromNode)(cb => this.server.stop(cb));
  }

  async inject(opts) {
    if (!this.server) await this.ready();

    return await (0, _bluebird.fromNode)(cb => {
      try {
        this.server.inject(opts, resp => {
          cb(null, resp);
        });
      } catch (err) {
        cb(err);
      }
    });
  }

  applyLoggingConfiguration(settings) {
    const config = _config2.default.withDefaultSchema(settings);
    const loggingOptions = (0, _configuration2.default)(config);
    const subset = {
      ops: config.get('ops'),
      logging: config.get('logging')
    };
    const plain = JSON.stringify(subset, null, 2);
    this.server.log(['info', 'config'], 'New logging configuration:\n' + plain);
    this.server.plugins['even-better'].monitor.reconfigure(loggingOptions);
  }
};
