import uiChrome from 'ui/chrome';
import pako from 'pako';

function bytesToHumanReadable(value) {
  if (value >= 1073741824) {
    value = "~" + (value / 1073741824).toFixed(2) + " GB";
  } else if (value >= 1048576) {
    value = "~" + (value / 1048576).toFixed(2) + " MB";
  } else if (value >= 1024) {
    value = "~" + (value / 1024).toFixed(2) + " KB";
  } else if (value > 1) {
    value = "~" + value + " bytes";
  } else if (value === 1) {
    value = "~" + value + " byte";
  } else {
    value = "0 bytes";
  };

  return value;
};

export const ENCODING_HEADER = 'content-encoding';
let total = 0;
let counter = 0;

function logIfLoggingEnabled(dataBefore, config, loggingEnabled) {
  if (loggingEnabled) {
    const url = config.url;
    const beforeSize = dataBefore.length;
    const afterSize = config.data.length;
    const savedDiff = beforeSize - afterSize;
    const compressionRatio = (beforeSize / afterSize).toFixed(2);
    total += savedDiff;
    counter++;
    console.log('-----------------------------------------------------------------------');
    console.log(`url: ${url}`);
    console.log('Payload before compression:');
    console.log(dataBefore);
    console.log('Stats:');
    console.log(`  before length: ${beforeSize}`);
    console.log(`  after length: ${afterSize}`);
    console.log(`  compression ratio: ${compressionRatio}`);
    console.log(`  saved: ${bytesToHumanReadable(savedDiff)}`);
    console.log('Global stats');
    console.log(`  request processed: ${counter}`);
    console.log(`  total saved: ${bytesToHumanReadable(total)}`);
    console.log('-----------------------------------------------------------------------');
  }
}

export function gzipInterceptor($q) {
  return {
    request: function (config) {
      const clientsideCompression = uiChrome.getInjected('clientsideCompression');
      if (!(clientsideCompression && clientsideCompression.compression_enabled)) {
        return config;
      }

      if (config.method && config.method.toLowerCase() === 'post') {
        if (typeof config.data === 'string' && config.data.length > clientsideCompression.compression_threshold) {
          const dataBefore = config.data;
          config.transformRequest = [];
          config.headers[ENCODING_HEADER] = 'gzip';
          config.data = pako.gzip(dataBefore);
          logIfLoggingEnabled(dataBefore, config, clientsideCompression.logging_enabled);
        } else if (typeof config.data === 'object' || config.data instanceof Array) {
          try {
            const dataBefore = JSON.stringify(config.data);
            if (dataBefore.length > clientsideCompression.compression_threshold) {
              config.transformRequest = [];
              config.headers[ENCODING_HEADER] = 'gzip';
              config.headers['content-type'] = 'application/json';
              config.data = pako.gzip(dataBefore);
              logIfLoggingEnabled(dataBefore, config, clientsideCompression.logging_enabled);
            } else {
              config.headers[ENCODING_HEADER] = undefined;
            }
          } catch (e) {
            config.headers[ENCODING_HEADER] = undefined;
          }
        } else {
          // delete is slow apparently, faster to set to undefined
          config.headers[ENCODING_HEADER] = undefined;
        }
      }
      return config;
    },
    requestError: function (rejection) {
      return $q.reject(rejection);
    },
    response: function (result) {
      return result;
    },
    responseError: function (response) {
      return $q.reject(response);
    }
  };
}
