'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createScrollStream = createScrollStream;

var _streams = require('./streams/streams');

var _config = require('../../../../server/formatters/config');

var _stream_helpers = require('./streams/common/stream_helpers');

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

async function createScrollStream(req, { fixedBatchSize } = {}) {
  const params = req.payload;
  const dataCluster = req.server.plugins.elasticsearch.getCluster('data');
  const client = (...args) => dataCluster.callWithRequest(req, ...args);

  const indexPatternId = params.indexPatternId;
  const kibanaIndex = req.server.config().get('kibana.index');
  const indexInfo = await (0, _stream_helpers.fetchIndexInfo)(client, kibanaIndex, indexPatternId);

  const adminCluster = req.server.plugins.elasticsearch.getCluster('admin');
  const adminClient = (...args) => adminCluster.callWithRequest(req, ...args);
  const frontendConfig = new _config.Config(adminClient, kibanaIndex);
  await frontendConfig.init();

  async function validateQuery() {
    const validationResponse = await client('indices.validateQuery', {
      body: {
        query: params.query
      },
      index: indexInfo.title,
      explain: true
    });

    if (!validationResponse.valid) {
      throw new Error(validationResponse.error);
    }
  }

  // Note: validation is ignored until the validate API is implemented in federate
  // (see https://github.com/sirensolutions/siren-platform/issues/1810)
  // await validateQuery();

  const debugLog = message => params.debug ? req.server.log(['info', 'siren_export'], message) : null;

  async function estimateAvgDocLength() {
    const sampleSize = 20;
    const source = (0, _stream_helpers.parseSource)(params._source);
    const fields = await (0, _stream_helpers.filterFields)(client, indexInfo.title, indexInfo.fields, source);
    const body = (0, _stream_helpers.buildRequestBody)(fields, source, params.query, params.sort);

    const sampleResponse = await client('siren_search', {
      filter_path: 'hits.total,hits.hits._source,hits.hits.fields,_scroll_id',
      body,
      index: indexInfo.title,
      size: sampleSize
    });
    const avgDocLength = JSON.stringify(sampleResponse).length / sampleSize;
    return avgDocLength;
  }

  let batchSize;
  if (!fixedBatchSize) {
    const avgDocLength = await estimateAvgDocLength();
    debugLog(`Average document length: ${avgDocLength}`);

    // Target batch length is ~15MB, not too small nor too big.
    const desiredBatchLength = 15 * 1024 * 1024;
    batchSize = Math.round(desiredBatchLength / avgDocLength);
  } else {
    batchSize = fixedBatchSize;
  }

  const maxBatchSize = 10000;
  batchSize = Math.min(batchSize, maxBatchSize);
  debugLog(`Calculated batch size: ${batchSize}`);

  const options = {
    docsToFetch: params.size,
    _source: params._source,
    normalize: params.format === 'csv' ? true : params.normalize,
    pretty: params.pretty,
    debug: params.debug,
    batchSize,
    objectMode: params.format === 'csv' ? true : false,
    sort: params.sort,
    omitNull: params.omitNull
  };

  const scrollStream = new _streams.ScrollStream(client, indexInfo, params.query, options);

  const errorMsg = (filename, format, err) => `Export of '${filename}.${format}' was aborted due to: ${err}`;

  scrollStream.on('log', message => req.server.log(['info', 'siren_export'], message));

  scrollStream.on('error', err => {
    req.server.log(['error', 'siren_export'], errorMsg(params.filename, params.format, err));
  });

  if (params.format !== 'csv') {
    return scrollStream;
  }

  const csvStream = new _streams.JSONToCSVStream(client, indexInfo, {
    _source: params._source,
    config: frontendConfig,
    formatted: params.formatted
  });

  csvStream.on('error', err => {
    req.server.log(['error', 'siren_export'], errorMsg(params.filename, params.format, err));
  });

  return scrollStream.pipe(csvStream);
}
