'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseSource = parseSource;
exports.fetchIndexInfo = fetchIndexInfo;
exports.filterFields = filterFields;
exports.buildRequestBody = buildRequestBody;
exports.getPathsForIndexPattern = getPathsForIndexPattern;

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _minimatch = require('minimatch');

var _minimatch2 = _interopRequireDefault(_minimatch);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function parseSource(_source) {
  const metaGlob = "_*";
  if (typeof _source === 'string') {
    return { includes: _source.split(','), excludes: [metaGlob] };
  } else if (typeof _source === 'boolean') {
    if (_source) {
      return { includes: '*', excludes: [metaGlob] };
    } else {
      return false;
    }
  } else if (_lodash2.default.isArray(_source)) {
    return { includes: _source, excludes: [metaGlob] };
  } else if (_lodash2.default.isPlainObject(_source)) {
    _source.excludes.push(metaGlob);
    return _source;
  }
}

async function fetchIndexInfo(client, kibanaIndex, indexPatternId) {
  const response = await client('get', {
    index: kibanaIndex,
    type: '_all',
    id: indexPatternId
  });
  return response._source['index-pattern'];
}

async function filterFields(client, indexName, fields, _source) {
  const indexMapping = await client('indices.getMapping', {
    index: indexName
  });
  const indexPatternPaths = getPathsForIndexPattern(indexMapping);
  const parsedFields = JSON.parse(fields);
  const filteredFields = parsedFields.filter(field => {
    const isMeta = field.name[0] === '_';
    const isNotMultifield = field.name in indexPatternPaths || field.scripted;
    const isIncluded = (0, _minimatch2.default)(field.name, `+(${_source.includes.join('|')})`);
    const isExcluded = (0, _minimatch2.default)(field.name, `+(${_source.excludes.join('|')})`);
    return !isMeta && isNotMultifield && isIncluded && !isExcluded;
  });

  return filteredFields;
}

function buildRequestBody(fields, _source, query, sort = []) {
  if (_lodash2.default.isPlainObject(sort)) {
    sort = [sort];
  };
  const scriptedFields = fields.filter(field => field.scripted);
  const sortMap = sort.reduce((acc, field) => {
    return Object.assign(acc, field);
  }, {});

  const scriptedFieldsToSort = scriptedFields.filter(field => field.name in sortMap).map(field => {
    const scriptedSort = {
      _script: {
        script: {
          lang: field.lang,
          inline: field.script
        },
        type: field.type
      }
    };
    Object.assign(scriptedSort._script, sortMap[field.name]);
    delete sortMap[field.name];

    return scriptedSort;
  });

  const nonscriptedSort = Object.keys(sortMap).map(fieldName => ({ [fieldName]: sortMap[fieldName] }));
  const mergedSort = nonscriptedSort.concat(scriptedFieldsToSort);

  let body = { _source, stored_fields: _source.includes, query, sort: mergedSort };
  const scriptFields = scriptedFields.reduce((acc, field) => {
    return _lodash2.default.set(acc, field.name, {
      script: {
        lang: field.lang,
        source: field.script
      }
    });
  }, {});
  body = _lodash2.default.isPlainObject(scriptFields) ? _lodash2.default.assign(body, { script_fields: scriptFields }) : body;
  return body;
}

function getPathsForIndexPattern(response) {
  const mapProperties = function (properties, paths = {}, currentPath = []) {
    _lodash2.default.each(properties, (property, fieldName) => {
      currentPath.push(fieldName);
      if (property.properties) {
        // nested
        mapProperties(property.properties, paths, currentPath);
      } else {
        paths[currentPath.join('.')] = _lodash2.default.clone(currentPath);
      }
      currentPath.pop();
    });
    return paths;
  };

  const paths = {};
  _lodash2.default.each(response, (mapping, indexName) => {
    // put the default type at the end
    (0, _lodash2.default)(mapping.mappings).sortBy((mapping, typeName) => typeName === '_default_' ? 1 : 0).each(mappings => {
      _lodash2.default.defaults(paths, mapProperties(mappings.properties));
    }).value();
  });
  return paths;
};
