'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JSONToCSVStream = undefined;

var _stream = require('stream');

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _stream_helpers = require('./common/stream_helpers');

var _type_map = require('../../../../../server/formatters/utils/type_map');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class JSONToCSVStream extends _stream.Transform {
  constructor(client, indexInfo, options = {}) {
    const defaultOpts = {
      _source: '*',
      writableObjectMode: true,
      writableHighWaterMark: 2,
      readableObjectMode: false
    };
    options = Object.assign(defaultOpts, options);
    super(options);
    this._client = client;
    this._indexInfo = indexInfo;
    this._fieldFormatMap = JSON.parse(indexInfo.fieldFormatMap || '{}');
    this._source = (0, _stream_helpers.parseSource)(options._source);
    this._batchCount = 0;
    this._frontendConfig = options.config;
    this._formatted = options.formatted;
    this._typeMap = this._frontendConfig ? new _type_map.TypeMap(this._frontendConfig) : {};
  }

  async _transform(data, encoding, callback) {
    try {
      let csv = '';
      if (this._batchCount === 0) {
        await this._init();
        csv = this._csvColumns.join(',') + '\n';
      }

      for (const hit of data) {
        const flattened = this._flattenHit(hit, true);
        csv += this._csvColumns.map(field => flattened.hasOwnProperty(field) ? this._stringifyField(flattened[field], field) : '').join(',') + '\n';
      }

      this._batchCount++;
      this.push(csv);

      callback();
    } catch (err) {
      this.destroy(err);
    }
  }

  async _init() {
    this._fields = await this._getFields();
    const deepTypes = ['geo_point', 'geo_shape'];
    this._csvColumns = Object.keys(this._fields).filter(field => !deepTypes.includes(this._fields[field].type)).sort();
  }

  async _getFields() {
    const filteredFields = await (0, _stream_helpers.filterFields)(this._client, this._indexInfo.title, this._indexInfo.fields, this._source);

    // helper function that 'expands' types with hidden fields (geo_shape, geo_point)
    const expandType = (type, childKeys) => {
      const matchingFields = filteredFields.filter(field => field.type === type);
      const expandedFields = _lodash2.default.flatten(matchingFields.map(field => childKeys.map(key => field.name + '.' + key)));
      return expandedFields;
    };

    const geoShapeKeys = ['type', 'coordinates', 'orientation', 'radius', 'geometries.type', 'geometries.coordinates', 'geometries.orientation', 'geometries.radius'];
    const geoShapeFields = expandType('geo_shape', geoShapeKeys);

    const geoPointKeys = ['lat', 'lon'];
    const geoPointFields = expandType('geo_point', geoPointKeys);

    const newFields = geoShapeFields.concat(geoPointFields).map(field => {
      return { name: field, type: 'object' };
    });
    const totalFields = filteredFields.concat(newFields);
    const fieldMap = {};
    for (const field of totalFields) {
      fieldMap[field.name] = field;
    }

    return fieldMap;
  }

  // pulled from _flatten_hit.js
  _flattenHit(hit, deep = false) {
    const flat = {};

    const flatten = (obj, keyPrefix) => {
      keyPrefix = keyPrefix ? keyPrefix + '.' : '';
      _lodash2.default.forOwn(obj, (val, key) => {
        key = keyPrefix + key;

        if (deep) {
          const isNestedField = this._fields[key] && this._fields[key].type === 'nested';
          const isArrayOfObjects = _lodash2.default.isArray(val) && _lodash2.default.isPlainObject(_lodash2.default.first(val));
          if (isArrayOfObjects && !isNestedField) {
            _lodash2.default.each(val, v => flatten(v, key));
            return;
          }
        } else if (flat[key] !== void 0) {
          return;
        }

        const hasValidMapping = this._fields[key] ? this._fields[key].type !== 'conflict' : true;
        const isValue = !_lodash2.default.isPlainObject(val);

        if (hasValidMapping && isValue) {
          if (!flat[key]) {
            flat[key] = val;
          } else if (_lodash2.default.isArray(flat[key])) {
            flat[key].push(val);
          } else {
            flat[key] = [flat[key], val];
          }
          return;
        }

        flatten(val, key);
      });
    };

    flatten(hit);
    return flat;
  }

  _stringifyField(val, field) {
    let fieldString;
    const fieldType = this._fieldFormatMap.hasOwnProperty(field) ? this._fieldFormatMap[field].id : this._fields[field].type;
    if (this._formatted && this._typeMap.hasConverter(fieldType)) {
      const TypeClass = this._typeMap.getType(fieldType);
      const params = this._fieldFormatMap.hasOwnProperty(field) ? this._fieldFormatMap[field].params : {};
      fieldString = new TypeClass(params).convert(val);
    } else if (_lodash2.default.isArray(val)) {
      fieldString = val.join(',');
    } else if (typeof val === 'string') {
      fieldString = val;
    } else {
      fieldString = JSON.stringify(val);
    }

    if (fieldString === '') {
      return '';
    }

    return `"${fieldString.replace(/\"/g, '""')}"`;
  }
}
exports.JSONToCSVStream = JSONToCSVStream;
