'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LicenseParser = undefined;

var _lodash = require('lodash');

class LicenseParser {
  constructor(federateResolver) {
    this.federateResolver = federateResolver;
  }

  /**
   * Verifies and transforms old license if valid
   * @param  {Object} license
   * @return {Object} license, new format
   */
  _verifyAndConvertOldLicense(license) {
    const requiredProps = ['license.isValid', 'license.content.graph-browser', 'license.content.valid-date'];
    if (!requiredProps.every(licenseProperty => (0, _lodash.has)(license, licenseProperty))) {
      if (license && (0, _lodash.get)(license, 'license.isValid') === false) {
        //No license found
        return {
          license_content: {
            description: 'Siren Community License',
            issue_date: '-',
            permissions: {
              federate: {
                max_concurrent_sessions: '1',
                max_nodes: '1'
              },
              investigate: {
                max_dashboards: '12',
                max_graph_nodes: '500',
                max_virtual_indices: '5'
              }
            },
            valid_date: '-'
          },
          license_validation: {
            is_valid: true
          }
        };
      }
      return license;
    }
    return {
      license_content: {
        permissions: {
          federate: {
            max_concurrent_sessions: '-1',
            max_nodes: '-1',
            session_timeout: '-1'
          },
          investigate: {
            max_dashboards: '-1',
            max_graph_nodes: license.license.content['graph-browser'] ? '-1' : 500,
            max_virtual_indices: '-1'
          }
        },
        valid_date: license.license.content['valid-date'],
        issue_date: license.license.content['issue-date']
      },
      license_validation: {
        is_valid: license.license.isValid
      }
    };
  }

  /**
   * Ensure the license is in latest format translates passed license if required.
   * severely malformed old licenses are left unchanged
   * @param  {Object} license
   * @return {Object} license is latest format
   */
  ensureLatest(license) {
    if (!this.federateResolver.checkApiAvailability('LICENSE_API_V2')) {
      return this._verifyAndConvertOldLicense(license);
    }
    return license;
  }

}

exports.LicenseParser = LicenseParser;
