import PropTypes from 'prop-types';
import React, { Component, Fragment } from 'react';
import { render, unmountComponentAtNode } from 'react-dom';
import { uiModules } from 'ui/modules';
import COMPATIBILITY_STATUS from '../../../common/utils/compatibility_status';

import {
  EuiToolTip
} from '@elastic/eui';

const compatibilityTooltip = function (key, compatibilityDefinition) {
  const toolTipContent = [];
  if (compatibilityDefinition.min_es) {
    toolTipContent.push(<span key={key}><b>Minimum Elasticsearch</b>: {compatibilityDefinition.min_es.toString()}</span>);
  }
  if (compatibilityDefinition.min_federate) {
    toolTipContent.push(<span key={key}><b>Minimum Federate</b>: {compatibilityDefinition.min_federate.toString()}</span>);
  }
  if (compatibilityDefinition.max_es) {
    toolTipContent.push(<span key={key}><b>Maximum Elasticsearch</b>: {compatibilityDefinition.max_es.toString()}</span>);
  }
  if (compatibilityDefinition.max_federate) {
    toolTipContent.push(<span key={key}><b>Minimum Federate</b>: {compatibilityDefinition.max_federate.toString()}</span>);
  }
  if (compatibilityDefinition.supported_es_exceptions) {
    toolTipContent.push(
      <span key={key}><b>Supported Elasticsearch (Exceptions)</b>: {compatibilityDefinition.supported_es_exceptions.toString()}</span>
    );
  }
  if (compatibilityDefinition.supported_federate_exceptions) {
    toolTipContent.push(
      <span key={key}><b>Supported Federate (Exceptions)</b>: {compatibilityDefinition.supported_federate_exceptions.toString()}</span>
    );
  }
  return (
    <Fragment>
      {toolTipContent}
    </Fragment>
  );
};

const decorateWithTooltip = function ({ key, object }) {
  return (<EuiToolTip position="right" title="Compatibility" content={compatibilityTooltip(key, object)}>
    <Fragment>{object.description}</Fragment>
  </EuiToolTip>);
};

class CompatibilityInfo extends Component {
  static getCompatibilityStatusText(status) {
    switch (status) {
      case COMPATIBILITY_STATUS.MIN_SUPPORTED:
        return { color: '#a9600a', text: 'Minimum Support' };
      case COMPATIBILITY_STATUS.FEATURE_COMPLETE:
        return { color: '#105A73', text: 'Feature Complete' };
      case COMPATIBILITY_STATUS.TARGET:
        return { color: '#017F75', text: 'Optimal' };
      default:
        return { color: 'red', text: 'Unknown' };
    }
  }

  constructor(props) {
    super(props);
    this.compatibilityStatus = CompatibilityInfo.getCompatibilityStatusText(this.props.compatibilityStatus);
    this.featuresDisabled = this.props.disabledFeatures.length > 0;
    this.disabledFeatureList = this.props.disabledFeatures.map(ele => {
      return {
        key: ele.key,
        title: ele.object.name || ele.key,
        description: decorateWithTooltip(ele)
      };
    });
    this.state = {
      tableExpanded: false
    };
  }

  toggle = () => {
    if (this.featuresDisabled) {
      this.setState(state => {
        state.tableExpanded = !state.tableExpanded;
        return state;
      });
    }
  }

  render() {
    let table;
    if (this.state.tableExpanded) {
      table = (
        <Fragment>
          <tr>
            <td className="title" colSpan="2" style={{ textAlign: 'center' }}>Disabled Features:</td>
          </tr>
          {
            this.disabledFeatureList.map(ele => {
              return (
                <tr key={ele.key}>
                  <td className="title">{ele.title}</td>
                  <td>{ele.description}</td>
                </tr>
              );
            })
          }
          <tr>
            <td className="title" colSpan="2" style={{ textAlign: 'center' }}>-----</td>
          </tr>
        </Fragment>
      );
    }
    return (
      <table className="table table-condensed kbn-settings-about-versions" style={{ marginBottom: '0px' }}>
        <tbody>
          <tr onClick={this.toggle} style={{ cursor: this.featuresDisabled ? 'pointer' : 'text' }}>
            <td className="title">Compatibility Status</td>
            <td>
              <span style={{ color: this.compatibilityStatus.color }}>
                {this.compatibilityStatus.text}
              </span>
            </td>
          </tr>
          {table}
        </tbody>
      </table>
    );
  }
}

CompatibilityInfo.propTypes = {
  disabledFeatures: PropTypes.arrayOf(
    PropTypes.shape({
      key: PropTypes.string.isRequired,
      object: PropTypes.shape({
        name: PropTypes.string.isRequired,
        description: PropTypes.string.isRequired
      }).isRequired
    })
  ).isRequired,
  compatibilityStatus: PropTypes.number.isRequired
};

uiModules.get('federate_resolver/core/directives')
  .directive('compatibilityInfo', function (federateResolver) {
    return {
      restrict: 'E',
      scope: {
        disabledFeatures: '='
      },
      link: function ($scope, $el) {
        const renderComponent = function () {
          render(
            (<CompatibilityInfo
              disabledFeatures={federateResolver.getDisabledFeatures()}
              compatibilityStatus={federateResolver.getCompatibilityStatus()}
            />), $el[0]
          );
        };
        renderComponent();
        $scope.$on('$destroy', () => unmountComponentAtNode($el[0]));
      }
    };
  });