'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 10.
 *
 * Looks for Kibi Word Cloud visualizations and upgrades them to Kibana 5 Tagcloud visualizations.
 */

class Migration10 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._server = configuration.server;
    this._type = 'visualization';
  }

  static get description() {
    return 'Migrate Kibi Word Cloud visualizations to Tag Cloud visualizations.';
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type);
    return objects.reduce((count, obj) => {
      const visState = JSON.parse(obj._source.visState);
      if (this._isUpgradeable(visState)) {
        return count + 1;
      }
      return count;
    }, 0);
  }

  /**
   * Checks if the specified visState described a Kibi Word Cloud
   * @param visState
   * @private
   */
  _isUpgradeable(visState) {
    return visState.type === 'kibi_wordcloud';
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type);
    if (objects.length === 0) {
      return 0;
    }
    let body = '';
    let count = 0;
    for (const obj of objects) {
      const visState = JSON.parse(obj._source.visState);
      if (this._isUpgradeable(visState)) {
        const newVisState = (0, _lodash.transform)(visState, (result, value, key) => {
          if (key === 'params') {
            const upgradedParams = {
              scale: 'linear',
              orientation: 'single',
              minFontSize: 18,
              maxFontSize: 72
            };
            result[key] = upgradedParams;
            return;
          } else if (key === 'type') {
            result[key] = 'tagcloud';
            return;
          } else if (key === 'aggs') {
            result.aggs = visState.aggs.map(agg => {
              const upgraded = (0, _lodash.cloneDeep)(agg);
              upgraded.enabled = true;
              if (upgraded.schema === 'bucket') {
                upgraded.schema = 'segment';
              }
              return upgraded;
            });
            return;
          }
          result[key] = value;
        });
        obj._source.uiStateJSON = '{}';
        obj._source.visState = JSON.stringify(newVisState);
        body += JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
        count++;
      }
    }

    if (count > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return count;
  }

}
exports.default = Migration10;
module.exports = exports['default'];
