'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _gremlin_server = require('../../../../../server/gremlin_server/gremlin_server');

var _gremlin_server2 = _interopRequireDefault(_gremlin_server);

var _gremlin_server_errors = require('../../../../../server/gremlin_server/gremlin_server_errors');

var _server_ontology_client = require('../../ontology/server_ontology_client');

var _server_ontology_client2 = _interopRequireDefault(_server_ontology_client);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 29.
 *
 * Looks for:
 *
 * - the relations inside ontology schema
 *
 * Then:
 *
 * - if there is merge join type, migrate it to automatic.
 */
class Migration29 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._index = configuration.config.get('kibana.index');

    this._ontologyType = 'ontology-model';
    this._ontologyId = 'default-ontology';
    this._gremlinStartTimeout = configuration.gremlinStartTimeout;
  }

  static get description() {
    return 'Migrate all merge join type relations to automatic join type.';
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const relations = await this._findAllUpgradeableRelations();
    return relations.length;
  }

  async _findAllUpgradeableRelations() {
    const gremlin = new _gremlin_server2.default(this._server, this._gremlinStartTimeout);
    const serverOntologyClient = new _server_ontology_client2.default(this._server, this._client, this._index, this._ontologyType, this._ontologyId);

    try {
      await gremlin.start('Migration 29');
      const mergeJoinRelations = [];

      const relations = await serverOntologyClient.getRelations();
      _lodash2.default.each(relations, function (relation) {
        if (relation.joinType === 'MERGE_JOIN') {
          mergeJoinRelations.push(relation);
        }
      });

      this._logger.info('All done stopping Siren Gremlin Server');
      await gremlin.stop('Migration 29 clean');
      return mergeJoinRelations;
    } catch (e) {
      this._logger.error('Migration 29 - count operation');
      if (e instanceof _gremlin_server_errors.GremlinStopError) {
        this._logger.error('Could not stop the Siren Gremlin Server for Migration 29');
        this._logger.error(e);
      } else if (e instanceof _gremlin_server_errors.GremlinError) {
        this._logger.error('Could not start the Siren Gremlin Server for Migration 29');
        this._logger.error(e);
        await gremlin.stop('Migration 29 error');
      } else {
        this._logger.error('Could not perform Migration 29');
        this._logger.error(e);
        await gremlin.stop('Migration 29 error');
      }
      throw e;
    }
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const migratedRelations = [];

    const objects = await this._findAllUpgradeableRelations();
    if (objects.length === 0) {
      return 0;
    }

    _lodash2.default.each(objects, obj => {
      if (obj.joinType === 'MERGE_JOIN') {
        this._logger.info(`Updating join type with _id=${obj._id}`);
        obj.joinType = null;
        migratedRelations.push(obj);
      }
    });

    const gremlin = new _gremlin_server2.default(this._server);
    const serverOntologyClient = new _server_ontology_client2.default(this._server, this._client, this._index, this._ontologyType, this._ontologyId);
    try {
      await gremlin.start('Migration 29');

      await serverOntologyClient.saveRelations(migratedRelations);

      await gremlin.stop('Migration 29 clean');
    } catch (e) {
      if (e instanceof _gremlin_server_errors.GremlinStopError) {
        this._logger.error('Could not stop the Siren Gremlin Server for Migration 29');
        this._logger.error(e);
      } else if (e instanceof _gremlin_server_errors.GremlinError) {
        this._logger.error('Could not start the Siren Gremlin Server for Migration 29');
        this._logger.error(e);
      } else {
        this._logger.error('Could not perform Migration 29');
        this._logger.error(e);
        await gremlin.stop('Migration 29 error');
      }
      throw e;
    }

    return migratedRelations.length;
  }
}
exports.default = Migration29;
module.exports = exports['default'];
