'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 34.
 *
 * Looks for default-ontology with missing title object in kibana.index
 *
 * if found set the title to Default Ontology
 *
 */

class Migration34 extends _migration2.default {

  constructor(configuration) {
    super(configuration);
    this._server = configuration.server;
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;

    this._defaultOntologyTitle = 'Default Ontology';
    this._id = 'ontology-model:default-ontology';
    this._type = 'doc';
    this._ontologyQuery = {
      query: {
        bool: {
          filter: [{
            term: {
              _id: this._id
            }
          }]
        }
      }
    };
  }

  static get description() {
    return 'Find default-ontology with missing title property and set it to Default Ontology';
  }

  _isUpgradeable(objects) {
    if (objects.length === 1 && objects[0]._source['ontology-model'] && objects[0]._source['ontology-model'].title === undefined) {
      return true;
    }
    return false;
  }

  async count() {
    try {
      const objects = await this.scrollSearch(this._index, this._type, this._ontologyQuery);
      if (this._isUpgradeable(objects)) {
        return 1;
      }
    } catch (e) {
      if (e.status !== 404) {
        this._logger.error('Unexpected error while fetching ontology', e);
      }
    }
    return 0;
  }

  async upgrade() {
    try {
      const objects = await this.scrollSearch(this._index, this._type, this._ontologyQuery);
      if (this._isUpgradeable(objects)) {
        await this._client.update({
          index: this._index,
          type: this._type,
          id: this._id,
          refresh: true,
          body: {
            doc: {
              'ontology-model': {
                title: this._defaultOntologyTitle
              }
            }
          }
        });
        return 1;
      }
    } catch (e) {
      if (e.status !== 404) {
        this._logger.error('Unexpected error while fetching ontology', e);
      }
    }
    return 0;
  }

}
exports.default = Migration34;
module.exports = exports['default'];
