'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _uuid = require('uuid');

var _uuid2 = _interopRequireDefault(_uuid);

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 39.
 *
 * Looks for dashboards with savedSearchId property:
 *
 * - removes the `savedSearchId`
 * - adds `coatJSON`
 * - bumps version to `2` as we changed the schema
 */

class Migration39 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._server = configuration.server;
    this._type = 'doc';
    this._query = {
      query: {
        bool: {
          filter: [{
            exists: {
              field: 'dashboard.savedSearchId'
            }
          }]
        }
      }
    };
    this._searchesQuery = {
      query: {
        bool: {
          filter: [{
            term: {
              type: 'search'
            }
          }]
        }
      }
    };
    this._indicesQuery = {
      query: {
        bool: {
          filter: [{
            term: {
              type: 'index-pattern'
            }
          }]
        }
      }
    };
    this._defaultViewSettings = {
      width: 1107,
      height: 400,
      zoom: 1.2,
      offsetX: 600,
      offsetY: 340
    };
    this._defaultNodeX = 300;
    this._defaultNodeY = 150;
    this._SINGLE_SEARCH = 'SINGLE_SEARCH';
    this._NO_SEARCH = 'NO_SEARCH';
  }

  static get description() {
    return 'Upgrade savedSearchId to coatJSON in dashboards, bump version to 3';
  }

  async count() {
    return await this.countHits(this._index, this._type, this._query);
  }

  async upgrade() {
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    if (objects.length === 0) {
      return 0;
    }

    // it is safe to put a mapping for coatJSON as it did not existed before
    await this._client.indices.putMapping({
      index: this._index,
      type: this._type,
      body: {
        properties: {
          dashboard: {
            properties: {
              coatJSON: {
                type: 'text'
              }
            }
          }
        }
      }
    });

    const searches = await this.scrollSearch(this._index, this._type, this._searchesQuery);
    const indexPatterns = await this.scrollSearch(this._index, this._type, this._indicesQuery);

    let body = '';
    for (const obj of objects) {
      const upgraded = this._upgradeObject(obj, searches, indexPatterns);
      body += JSON.stringify({
        index: {
          _index: obj._index,
          _type: obj._type,
          _id: obj._id
        }
      }) + '\n' + JSON.stringify(upgraded) + '\n';
    }
    await this._client.bulk({
      refresh: true,
      body: body
    });
    return objects.length;
  }

  _getUUID() {
    return _uuid2.default.v1();
  }

  _shouldSetUseGlobalTimeFilter(searchId, searches, indexPatterns) {
    const search = (0, _lodash.find)(searches, '_id', searchId);
    if (!search) {
      this._logger.warning('Missing saved search object: ' + searchId);
      return false;
    }
    const searchSource = JSON.parse(search._source.search.kibanaSavedObjectMeta.searchSourceJSON);
    const indexPattern = (0, _lodash.find)(indexPatterns, '_id', searchSource.index);
    if (!indexPattern) {
      this._logger.warning('Missing index-pattern object: ' + searchSource.index);
      return false;
    }

    return !!indexPattern._source['index-pattern'].timeFieldName;
  }

  /**
   * Upgrades a single object.
   *
   * @param {Object} obj The object to upgrade.
   */
  _upgradeObject(obj, searches, indexPatterns) {
    if (obj._source.dashboard) {
      if (obj._source.dashboard.savedSearchId) {
        const coatJSON = {
          items: [{
            id: this._getUUID(),
            type: 'node',
            d: {
              isRoot: true,
              entity: {
                id: obj._source.dashboard.savedSearchId
              }
            },
            x: this._defaultNodeX,
            y: this._defaultNodeY
          }],
          datamodelType: this._SINGLE_SEARCH,
          viewSettings: this._defaultViewSettings
        };
        if (this._shouldSetUseGlobalTimeFilter(obj._source.dashboard.savedSearchId, searches, indexPatterns)) {
          coatJSON.items[0].d.entity.useGlobalTimeFilter = true;
        }
        obj._source.dashboard.coatJSON = JSON.stringify(coatJSON);
      } else {
        obj._source.dashboard.coatJSON = JSON.stringify({
          datamodelType: this._NO_SEARCH
        });
      }
      obj._source.dashboard.version = 3;
      delete obj._source.dashboard.savedSearchId;
    }
    return obj._source;
  }
}
exports.default = Migration39;
module.exports = exports['default'];
