'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _requestPromise = require('request-promise');

var _requestPromise2 = _interopRequireDefault(_requestPromise);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class ServerOntologyClient {

  constructor(server, esClient, index, ontologyType, ontologyId) {
    const config = server.config();
    const sslCA = config.get('investigate_core.gremlin_server.ssl.ca');
    if (sslCA) {
      this.sslCA = _fs2.default.readFileSync(sslCA);
    }
    this._gremlinURL = config.get('investigate_core.gremlin_server.url');
    this._client = esClient;
    this._ontologyIndex = index;
    this._ontologyType = ontologyType;
    this._ontologyId = ontologyId;
  }

  schema(path, options) {
    const opts = {
      method: options.method ? options.method : 'POST',
      uri: options.url + path,
      body: options.data ? options.data : '{}',
      json: true
    };

    if (this.sslCA) {
      opts.ca = this.sslCA;
    }
    return (0, _requestPromise2.default)(opts);
  }

  _encodeUrl(str) {
    return encodeURIComponent(str).replace(/!/g, '%21').replace(/'/g, '%27').replace(/\(/g, '%28').replace(/\)/g, '%29').replace(/\*/g, '%2A').replace(/%20/g, '+');
  }

  _updateOntology(model) {
    if (this._ontologyType === 'doc') {
      return this._client.update({
        index: this._ontologyIndex,
        type: this._ontologyType,
        id: this._ontologyId,
        body: {
          doc: {
            'ontology-model': {
              model: model
            }
          }
        },
        refresh: true
      });
    } else {
      return this._client.update({
        index: this._ontologyIndex,
        type: this._ontologyType,
        id: this._ontologyId,
        body: {
          doc: {
            model: model
          }
        },
        refresh: true
      });
    }
  }

  getRelations() {
    const opts = {
      method: 'GET',
      url: this._gremlinURL
    };
    return this.schema('/schema/relations', opts);
  }

  getEntities() {
    const opts = {
      method: 'GET',
      url: this._gremlinURL
    };
    return this.schema('/schema/entities', opts).then(entities => {
      (0, _lodash.each)(entities, function (entity) {
        if (entity.label) {
          entity.label = entity.label.substring(0, entity.label.lastIndexOf('@'));
        }
      });
      return entities;
    });
  }

  createEntity(entity) {
    const opts = {
      method: 'POST',
      data: entity,
      url: this._gremlinURL
    };

    return this.schema('/schema/entity/' + this._encodeUrl(entity.id), opts).then(returnedModel => {
      return this._updateOntology(returnedModel);
    });
  }

  deleteEntity(entityId) {
    const opts = {
      method: 'DELETE',
      url: this._gremlinURL
    };
    return this.schema('/schema/entity/' + this._encodeUrl(entityId), opts).then(returnedModel => {
      return this._updateOntology(returnedModel);
    });
  }

  deleteEntityRelations(entityId) {
    const opts = {
      method: 'DELETE',
      url: this._gremlinURL
    };
    return this.schema('/schema/relationByDomainOrRange/' + this._encodeUrl(entityId), opts).then(returnedModel => {
      return this._updateOntology(returnedModel);
    });
  }

  saveRelations(relations) {
    const opts = {
      method: 'POST',
      data: relations,
      url: this._gremlinURL
    };

    return this.schema('/schema/relations', opts).then(returnedModel => {
      return this._updateOntology(returnedModel);
    });
  }
}

exports.default = ServerOntologyClient;
module.exports = exports['default'];
