import 'plugins/investigate_core/management/sections/kibi_datasources/styles/datasources_editor.less';
import 'plugins/investigate_core/management/sections/kibi_datasources/services/_saved_datasource';
import 'plugins/investigate_core/management/sections/kibi_datasources/services/saved_datasources';
import 'ui/kibi/components/query_engine_client/query_engine_client';
import 'ui/kibi/directives/kibi_validate';
import SetDatasourceSchemaProvider from 'plugins/investigate_core/management/sections/kibi_datasources/lib/set_datasource_schema';
import { DatasourceTypes, DatasourceTypeValues } from 'plugins/investigate_core/management/sections/kibi_datasources/lib/datasources';
import template from 'plugins/investigate_core/management/sections/kibi_datasources/index.html';
import kibiUtils from 'kibiutils';
import uiRoutes from 'ui/routes';
import { uiModules } from 'ui/modules';
import { map, noop } from 'lodash';
import { jdbcDatasourceTranslate } from 'plugins/investigate_core/management/sections/kibi_datasources/services/jdbc_datasource_translate';

uiRoutes
  .when('/management/siren/datasources', {
    template,
    reloadOnSearch: false,
    resolve: {
      datasource: function (savedDatasources) {
        return savedDatasources.get();
      },
      isNew: function () {
        return true;
      }
    }
  })
  .when('/management/siren/datasources/:id?', {
    template,
    reloadOnSearch: false,
    resolve: {
      datasource: function ($route, courier, savedDatasources, jdbcDatasources) {
      // first try to get it from _siren/connector
        return jdbcDatasources.get($route.current.params.id)
          .then(datasource => {
            return jdbcDatasourceTranslate.datasourceToSavedDatasource(datasource);
          })
          .catch(err => {
            return savedDatasources.get($route.current.params.id)
              .catch(courier.redirectWhenMissing({
                datasource: '/management/siren/datasources'
              }));
          });
      },
      isNew: function () {
        return false;
      }
    }
  });

function controller(Private, $window, $scope, $route, kbnUrl, createNotifier,
  queryEngineClient, $element, kibiWarnings, jdbcDatasources, confirmModal, federateResolver) {
  const setDatasourceSchema = Private(SetDatasourceSchemaProvider);
  const notify = createNotifier({
    location: 'Datasources Configuration Editor'
  });
  const datasource = $scope.datasource = $route.current.locals.datasource;

  $scope.DatasourceTypes = DatasourceTypes;
  // Setup parameters for connection helper panel
  $scope.databaseParams = {
    databaseType: '',
    databaseName: ''
  };

  // siren: we should initialize it if there is no advanced parameters
  if (!datasource.datasourceParams.properties) {
    datasource.datasourceParams.properties = [];
  }
  if (federateResolver.isAvailable('REMOTE_ES_CONNECTOR')) {
    $scope.remoteEsAvailable = true;
    jdbcDatasources.getRemoteClusters()
      .then(clusters => {
        $scope.remoteClusters = clusters;
      })
      .catch(notify.error);
  } else {
    $scope.remoteEsAvailable = false;
  }

  // Default values for JDBC datasource params
  jdbcDatasources.getDatasourceTypes()
    .then(res => {
      // Pull out the connection types to populate the dropdown select in the connection helper
      $scope.datasourceDefaults = res;
      $scope.possibleDatabaseTypes = Object.keys($scope.datasourceDefaults);
    })
    .catch(notify.error);

  $scope.isNew = $route.current.locals.isNew;
  $scope.isValid = function () {
    return $element.find('form[name="objectForm"]').hasClass('ng-valid');
  };

  $scope.isDeleteValid = function () {
    return !$scope.isNew && $scope.datasource &&
    (datasource.datasourceType === 'rest' || DatasourceTypeValues.has(datasource.datasourceType));
  };

  $scope.deleteObject = function () {
    // ask user to delete it as well
    const confirmModalOptions = {
      confirmButtonText: 'Delete the datasource',
      onConfirm: () => {
        if (DatasourceTypeValues.has(datasource.datasourceType)) {
          const id = $scope.datasource.title;
          jdbcDatasources.delete(id).then(() => {
            notify.info(`Datasource ${id} successfully deleted`);
            kbnUrl.change('management/siren/datasources');
          }).catch(err => {
            notify.error(err);
          });
        } else {
          datasource.delete().then(function () {
            notify.info('Datasource ' + datasource.title + ' successfully deleted');
            kbnUrl.change('management/siren/datasources');
          }).catch(notify.error);
        }
      }
    };
    confirmModal(
      `Are you sure you want to delete the datasource ${datasource.title}?`,
      confirmModalOptions
    );
  };

  // NOTE: this method must always retuen a Promise
  $scope.saveObject = function () {
    const saveJdbcDatasource = function () {
      const d = jdbcDatasourceTranslate.savedDatasourceToJdbcDatasource(datasource);
      return jdbcDatasources.save(d)
        .then(() => {
          notify.info('Datasource ' + d._id + ' successfully saved');
          kbnUrl.change('management/siren/datasources/' + d._id);
        })
        .catch(notify.error);
    };

    if (DatasourceTypeValues.has(datasource.datasourceType)) {

      return jdbcDatasources.get(datasource.title)
        .then(function () {

          const overwriteModalOptions = {
            title: 'Datasource exists!',
            confirmButtonText: 'Yes, overwrite',
            cancelButtonText: 'No',
            onConfirm: saveJdbcDatasource,
            onCancel: noop
          };
          confirmModal(
            'Datasource with id: ' + datasource.title + ' exists. Do you want to overwrite?',
            overwriteModalOptions
          );
        }).catch(err => {
          if (err.status === 404) {
            return saveJdbcDatasource();
          }
          notify.error(err);
        });
    }

    const _save = function () {
      // old jdbc datasources
      if (kibiUtils.isJDBC(datasource.datasourceType)) {
        const msg = 'Changes in a JDBC datasource requires the application to be restarted. ' +
          'Please restart Siren Investigate and do not forget to set investigate_core.load_jdbc to true.';
        notify.warning(msg);
      }
      return _saveDatasource(datasource);
    };

    if (kibiWarnings.datasource_encryption_warning) {
      let encrypted = false;
      for (let s = 0; s < datasource.schema.length; s++) {
        const field = datasource.schema[s];
        if (field.encrypted) {
          encrypted = true;
          break;
        }
      }
      if (encrypted) {
        return new Promise(function (resolve, reject) {
          const confirmDatasourceSaveModalOptions = {
            confirmButtonText: 'Save this datasource',
            onConfirm: () => {
              _save()
                .then(resolve)
                .catch(reject);
            }
          };
          confirmModal(
            'You haven\'t set a custom encryption key;' +
            ' are you sure you want to save this datasource?',
            confirmDatasourceSaveModalOptions
          );
        });
      } else {
        return _save();
      }
    } else {
      return _save();
    };
  };

  /** _populateConnectionString
   *  databaseParams {object}
   *  databaseParams.databaseType {string} The value selected in the connection helper dropdown select
   *  databaseParams.databaseName {string} (optional) User entered database name for JDBC connection
   *
   * Takes in database helper parameters and populates the suggested connection string.
   * Also populates the drivername parameter if not already set by the user.
  */
  function _populateConnectionString(databaseParams) {
    if (datasource.datasourceType === DatasourceTypes.JDBC && databaseParams.databaseType) {
      if ($scope.isNew) {
        const databaseName = databaseParams.databaseName || '';
        const defaultPort = $scope.datasourceDefaults[databaseParams.databaseType].defaultPort || '';
        // Pull out the default driver class names
        const defaultDriverClassNames = map($scope.datasourceDefaults, defaultObject => defaultObject.driverClassName);

        // if there is no drivername (or the drivername is one of the defaults)
        // Update it with the new default. If it is custom-entered by the user, leave it there
        if (!datasource.datasourceParams.drivername) {
          datasource.datasourceParams.drivername = $scope.datasourceDefaults[databaseParams.databaseType].driverClassName;
        } else if (defaultDriverClassNames.indexOf(datasource.datasourceParams.drivername) !== -1) {
          datasource.datasourceParams.drivername = $scope.datasourceDefaults[databaseParams.databaseType].driverClassName;
        }

        datasource.datasourceParams.disclaimer = $scope.datasourceDefaults[databaseParams.databaseType].disclaimer || '';
        let url = $scope.datasourceDefaults[databaseParams.databaseType].defaultURL;

        if (url) {
          // SQL Server and Dremio add the database as query params
          let databaseString;
          if (databaseParams.databaseType === 'SQL Server 2017') {
            databaseString = `;databaseName=${databaseName}`;
          } else if (databaseParams.databaseType === 'Dremio') {
            databaseString = `;schema=${databaseName}`;
          } else {
            databaseString = `/${databaseName}`;
          }

          url = url.replace(/{{port}}/, (defaultPort) ? defaultPort : '');
          url = url.replace(/{{host}}/, 'localhost');

          if (databaseParams.databaseType === 'Impala' && databaseName) {
            url = url.replace(/\/default/, `/${databaseName}`);
          } else {
            url = url.replace(/{{databasename}}/, (databaseName) ? databaseString : '');
          }
        }
        datasource.datasourceParams.connection_string = url;
      }
    }
  }

  function _saveDatasource(datasource) {
    // make sure that any parameter which does not belong to the schema
    // is removed from datasourceParams
    for (const prop in datasource.datasourceParams) {
      if (datasource.datasourceParams.hasOwnProperty(prop)) {
        let remove = true;
        for (let j = 0; j < datasource.schema.length; j++) {
          if (datasource.schema[j].name === prop) {
            remove = false;
            break;
          }
        }
        if (remove) {
          delete datasource.datasourceParams[prop];
        }
      }
    }

    return datasource.save().then(function (datasourceId) {
      if (datasourceId) {
        $scope.isNew = false;

        notify.info('Datasource ' + datasource.title + ' successfully saved');
        queryEngineClient.clearCache().then(function () {
          kbnUrl.change('management/siren/datasources/' + datasourceId);
        });
      }
    });
  }

  // Toggle the connection helper panel
  $scope.openConnectionPanel = () => $scope.toggleConnectionPanel = !$scope.toggleConnectionPanel;
  $scope.acceptConnectionString = () => $scope.toggleConnectionPanel = false;

  $scope.newObject = function () {
    kbnUrl.change('management/siren/datasources', {});
  };

  $scope.$watch('datasource.datasourceType', function (newval, oldval) {
    // here reinit the datasourceDef
    if (DatasourceTypeValues.has(datasource.datasourceType) && datasource.title === 'New Saved Datasource') {
      datasource.title = '';
    }

    setDatasourceSchema(datasource);
  });

  $scope.$watchGroup([
    'databaseParams.databaseType',
    'databaseParams.databaseName'
  ], function ([ databaseType, databaseName ]) {
    _populateConnectionString({
      databaseType,
      databaseName
    });
  });

  // currently supported only for sql_jdbc_new
  $scope.testConnection = function (showModal) {
    const title = (datasource.datasourceType ===  DatasourceTypes.JDBC ? 'JDBC' : 'Remote Elasticsearch')
    + ' datasource configuration successful';
    const modalOptions = {
      title: title,
      confirmButtonText: 'Yes, take me there',
      cancelButtonText: 'No, will do later',
      onConfirm: () => kbnUrl.change('/management/siren/virtualindices/'),
      onCancel: () => {}
    };

    jdbcDatasources.validate(jdbcDatasourceTranslate.savedDatasourceToJdbcDatasource(datasource))
      .then(res => {
        notify.info('Connection OK');
        showModal && confirmModal(
          'Next step is to map a remote table (or view) by creating a Virtual Index. Do that now?',
          modalOptions
        );
      })
      .catch(err => {
        if (err && err.error && err.error.reason) {
          notify.error(err.error.reason + ' ' + JSON.stringify(err));
        } else if (err && err.error && err.message) {
          notify.error(err.message + ' ' + JSON.stringify(err));
        } else if (err) {
          notify.error('Unknown error: ' + JSON.stringify(err));
        } else {
        // this happens when Investigate backend is down
          notify.error('Unknown connection error. Please refresh the browser and try again');
        }
      });
  };

  // expose some methods to the navbar buttons
  [ 'isValid', 'newObject', 'saveObject', 'isDeleteValid', 'deleteObject' ]
    .forEach(name => {
      $element.data(name, $scope[name]);
    });
}

uiModules
  .get('apps/management', ['kibana'])
  .controller('DatasourcesEditor', controller);
