'use strict';

var _migration_runner = require('kibiutils/lib/migrations/migration_runner');

var _migration_runner2 = _interopRequireDefault(_migration_runner);

var _migration_logger = require('kibiutils/lib/migrations/migration_logger');

var _migration_logger2 = _interopRequireDefault(_migration_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * The migrations plugin checks if there are objects that can be upgraded.
 */
module.exports = function (kibana) {

  return new kibana.Plugin({
    // kibi: if no federate plugin found, migrations should not be run at all.
    // Otherwise, we receive an exception when Gremlin server looking for index patterns
    require: ['elasticsearch', 'siren_federate', 'saved_objects_api', 'query_engine'],
    id: 'migrations',
    init: async function (server, options, kbnServer) {
      this.status.yellow('Checking for out of date objects.');
      const checkMigrations = async () => {
        const logger = new _migration_logger2.default(server, 'migrations');
        if (server._migrationChecksDone === true) {
          // skip it was already done
          server.log(['info', 'migration'], 'Skipping migrations check - done already');
          return;
        }

        const runner = new _migration_runner2.default(server, logger, kbnServer);
        const cluster = server.plugins.elasticsearch.getCluster('admin');
        let count;
        try {
          count = await runner.count();
        } catch (err) {
          server.log(['error', 'migration'], err);
          this.status.red(`An error occurred while checking for out of date objects: ${err}`);
        }

        if (count > 0) {
          const results = await cluster.callWithInternalUser('count', {
            index: server.config().get('kibana.index')
          });

          if (results.count === 0) {
            this.status.red("Migrations will be run on empty siren index automatically");
            try {
              await runner.upgrade();
              this.status.green("Initial migrations were completed");
            } catch (err) {
              this.status.red(err);
            };
          } else {
            const eS6ConfigCheckPromise = cluster.callWithInternalUser('mget', {
              ignore: [404],
              body: {
                docs: [{
                  _index: server.config().get('kibana.index'),
                  _type: 'doc',
                  _id: 'config:siren'
                }]
              }
            });

            const eS5ConfigCheckPromise = cluster.callWithInternalUser('mget', {
              ignore: [404],
              body: {
                docs: [{
                  _index: server.config().get('kibana.index'),
                  _type: 'config',
                  _id: 'siren'
                }]
              }
            });

            const configChecks = await Promise.all([eS5ConfigCheckPromise, eS6ConfigCheckPromise]);
            const found = configChecks.some(result => result.docs.some(doc => doc.found));
            if (!found) {
              this.status.red('No config object found found, migrating to create new one.');
              try {
                await runner.upgrade();
                this.status.green('Config created');
              } catch (err) {
                server.log(['error', 'migration'], err);
                this.status.red(err);
              }
            } else {
              this.status.red(`There are ${count} outdated or missing objects. Please run "bin/investigate upgrade" and restart the server.`);
            }
          }
        } else {
          this.status.green('All objects are up to date.');
          server._migrationChecksDone = true;
        }
      };

      const status = server.plugins.elasticsearch.status;
      if (status && status.state === 'green') {
        await checkMigrations();
      } else {
        status.on('change', async () => {
          // eslint-disable-line memoryleaks
          if (server.plugins.elasticsearch.status.state === 'green') {
            await checkMigrations();
          }
        });
      }
    }
  });
};
