'use strict';

var _symbols = require('../_symbols');

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _url = require('url');

var _url2 = _interopRequireDefault(_url);

var _requestPromise = require('request-promise');

var _requestPromise2 = _interopRequireDefault(_requestPromise);

var _jsonpath = require('jsonpath');

var _jsonpath2 = _interopRequireDefault(_jsonpath);

var _abstract_query = require('./abstract_query');

var _abstract_query2 = _interopRequireDefault(_abstract_query);

var _query_helper = require('../query_helper');

var _query_helper2 = _interopRequireDefault(_query_helper);

var _rules_helper = require('../rules_helper');

var _rules_helper2 = _interopRequireDefault(_rules_helper);

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function RestQuery(server, queryDefinition, cache) {
  _abstract_query2.default.call(this, server, queryDefinition, cache);
  this.logger = (0, _logger2.default)(server, 'rest_query');
  this.queryHelper = new _query_helper2.default(server);
  this.rulesHelper = new _rules_helper2.default(server);
}

RestQuery.prototype = _lodash2.default.create(_abstract_query2.default.prototype, {
  'constructor': RestQuery
});

/*
 * Return a promise which when resolved should return true or false.
 */
RestQuery.prototype.checkIfItIsRelevant = function (options) {
  if (this._checkIfSelectedDocumentRequiredAndNotPresent(options)) {
    this.logger.warn('No elasticsearch document selected while required by the REST query. [' + this.config.id + ']');
    return _bluebird2.default.resolve(_symbols.SELECTED_DOCUMENT_NEEDED);
  }

  // no document selected there is nothing to check against
  if (!options.selectedDocuments || options.selectedDocuments.length === 0) {
    return _bluebird2.default.resolve(_symbols.QUERY_RELEVANT);
  }

  // empty rules - let it go
  if (this.config.activation_rules.length === 0) {
    return _bluebird2.default.resolve(_symbols.QUERY_RELEVANT);
  }

  // evaluate the rules
  return this.rulesHelper.evaluate(this.config.activation_rules, options).then(res => res ? _symbols.QUERY_RELEVANT : _symbols.QUERY_DEACTIVATED);
};

RestQuery.prototype._logFailedRequestDetails = function (msg, originalError, resp) {
  this.logger.error(msg, originalError);
  this.logger.error('See the full resp object below');
  this.logger.error(resp);
};

const mergeObjects = function (dest, sourceObject, sourcePath) {
  const source = _lodash2.default.get(sourceObject, sourcePath);
  if (source) {
    _lodash2.default.each(source, candidate => {
      const found = _lodash2.default.find(dest, c => c.name === candidate.name);
      if (!found) {
        dest.push(candidate);
      }
    });
  }
};

RestQuery.prototype.fetchResults = function (options, onlyIds, idVariableName) {
  const self = this;

  const urlS = this.config.datasource.datasourceClazz.datasource.datasourceParams.url;
  const timeout = this.config.datasource.datasourceClazz.datasource.datasourceParams.timeout;
  const maxAge = this.config.datasource.datasourceClazz.datasource.datasourceParams.max_age;
  const username = this.config.datasource.datasourceClazz.datasource.datasourceParams.username;
  const password = this.config.datasource.datasourceClazz.datasource.datasourceParams.password;
  const cacheEnabled = this.config.datasource.datasourceClazz.datasource.datasourceParams.cache_enabled;

  return new _bluebird2.default(function (fulfill, reject) {
    const start = new Date().getTime();
    const regex = /^GET|POST$/;

    if (!regex.test(self.config.rest_method)) {
      reject(new Error('Only GET|POST methods are supported at the moment'));
      return;
    }

    if (!(self.config.rest_params instanceof Array)) {
      reject(new Error('rest_params should be an Array. Check the elasticsearch mapping'));
      return;
    }

    if (!(self.config.rest_headers instanceof Array)) {
      reject(new Error('rest_headers should be an Array. Check the elasticsearch mapping'));
      return;
    }

    // user can also use a special variables like $auth_token
    const availableVariables = {
      // for now we support only auth_token username password
      // so user can provide any of these in params, headers, or body
      '${auth_token}': self.config.datasource.datasourceClazz.populateParameters('${auth_token}'),
      '${username}': self.config.datasource.datasourceClazz.populateParameters('${username}'),
      '${password}': self.config.datasource.datasourceClazz.populateParameters('${password}')
    };

    // get all params from datasource and merge them with the one from the query
    const mergedHeaders = [];
    const mergedParams = [];
    mergeObjects(mergedHeaders, self.config, 'rest_headers');
    mergeObjects(mergedHeaders, self.config, 'datasource.datasourceParams.headers');
    mergeObjects(mergedParams, self.config, 'rest_params');
    mergeObjects(mergedParams, self.config, 'datasource.datasourceParams.params');

    // the whole replacement of values is happening here
    self.queryHelper.replaceVariablesForREST(mergedHeaders, mergedParams, self.config.rest_body, self.config.rest_path, options, availableVariables).then(function (results) {
      // here convert the params and headers from array to map
      const headers = _lodash2.default.zipObject(_lodash2.default.map(results.headers, 'name'), _lodash2.default.map(results.headers, 'value'));
      const params = _lodash2.default.zipObject(_lodash2.default.map(results.params, 'name'), _lodash2.default.map(results.params, 'value'));
      const body = results.body;
      const path = results.path;

      let key;
      if (self.cache && cacheEnabled) {
        key = self.generateCacheKey(self.config.rest_method, urlS, path, JSON.stringify(headers), JSON.stringify(params), body, self._getUsername(options));
        const v = self.cache.get(key);
        if (v) {
          return fulfill(v);
        }
      }

      // to check any option visit
      // https://github.com/request/request#requestoptions-callback
      const rpOptions = {
        method: self.config.rest_method,
        uri: _url2.default.parse(_url2.default.resolve(urlS, path)),
        headers: headers,
        timeout: timeout || 5000,
        transform: function (body, resp) {
          let msg;
          const data = {
            queryId: self.id,
            label: self.config.label,
            results: {}
          };
          if (resp.statusCode !== self.config.rest_resp_status_code) {
            msg = 'Invalid response status code: [' + resp.statusCode + '] Expected: [' + self.config.rest_resp_status_code + ']';
            self._logFailedRequestDetails(msg, null, resp);
            throw new Error(msg);
          }

          // TODO: / Kibi / change this once we support xml resp or text resp
          let json;
          try {
            json = JSON.parse(body);
          } catch (e) {
            msg = 'Error while parsing body as JSON. Details: ' + e.message;
            self._logFailedRequestDetails(msg, e, resp);
            throw new Error(msg);
          }

          data.results = json;

          if (idVariableName && self.config.rest_variables) {
            const o = _lodash2.default.find(self.config.rest_variables, function (v) {
              return v.name === idVariableName;
            });

            if (o) {
              try {
                data.ids = _jsonpath2.default.query(json, o.value);
              } catch (e) {
                msg = 'Error while executing the JSONPath expressionXX. Details: ' + e.message;
                self._logFailedRequestDetails(msg, e, resp);
                throw new Error(msg);
              }
            }
          }

          if (self.cache && cacheEnabled) {
            self.cache.set(key, data, maxAge);
          }
          return data;
        }
      };

      if (username && password) {
        rpOptions.auth = {
          // as they might be encrypted make sure to call populateParameters
          username: self.config.datasource.datasourceClazz.populateParameters('${username}'),
          password: self.config.datasource.datasourceClazz.populateParameters('${password}'),
          sendImmediately: false
        };
      }

      if (self.config.rest_method === 'GET') {
        rpOptions.qs = params;
      } else if (self.config.rest_method === 'POST') {
        rpOptions.body = body;
        // WARNING: do not set rpOptions.json = true/false; even for json content
        // rather ask user to set correct Content-Type: application/json header
      }

      return (0, _requestPromise2.default)(rpOptions).then(function (resp) {
        fulfill(resp);
      }).catch(function (err) {
        const msg = 'Rest request failed: ' + JSON.stringify(rpOptions.uri, null, ' ') + '.\nDetails: ' + err.message;
        self._logFailedRequestDetails(msg, err, null);
        reject(new Error(msg));
      });
    });
  });
};

RestQuery.prototype._postprocessResults = function (data) {
  return data;
};

module.exports = RestQuery;
