'use strict';

var _symbols = require('./_symbols');

var _symbols2 = _interopRequireDefault(_symbols);

var _kibiutils = require('kibiutils');

var _kibiutils2 = _interopRequireDefault(_kibiutils);

var _requestPromise = require('request-promise');

var _requestPromise2 = _interopRequireDefault(_requestPromise);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _lruCache = require('lru-cache');

var _lruCache2 = _interopRequireDefault(_lruCache);

var _logger = require('./logger');

var _logger2 = _interopRequireDefault(_logger);

var _set_datasource_clazz = require('./datasources/set_datasource_clazz');

var _set_datasource_clazz2 = _interopRequireDefault(_set_datasource_clazz);

var _rest_query = require('./queries/rest_query');

var _rest_query2 = _interopRequireDefault(_rest_query);

var _error_query = require('./queries/error_query');

var _error_query2 = _interopRequireDefault(_error_query);

var _inactivated_query = require('./queries/inactivated_query');

var _inactivated_query2 = _interopRequireDefault(_inactivated_query);

var _missing_selected_document_query = require('./queries/missing_selected_document_query');

var _missing_selected_document_query2 = _interopRequireDefault(_missing_selected_document_query);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function QueryEngine(server) {
  this.server = server;
  this.config = server.config();
  const sslCA = this.config.get('investigate_core.gremlin_server.ssl.ca');
  if (sslCA) {
    this.sslCA = _fs2.default.readFileSync(sslCA);
  }
  this.queries = [];
  this.initialized = false;
  this.log = (0, _logger2.default)(server, 'query_engine');
  // kibi: we are using admin cluster directly where we don't have a access to the request object,
  // for example, data sources and templates need to be loaded before user logged in the system
  this.cluster = server.plugins.elasticsearch.getCluster('admin');

  this.savedObjectsClient = server.savedObjectsClientFactory({
    callCluster: this.cluster.callWithInternalUser
  });

  this.savedObjectsAPI = server.plugins.saved_objects_api;
}

QueryEngine.prototype._onStatusGreen = function () {
  return this.loadPredefinedData().then(() => {
    return this.reloadQueries().then(() => {
      this.initialized = true;
      return true;
    });
  });
};

QueryEngine.prototype._init = function (cacheSize = 500, enableCache = true, cacheMaxAge = 1000 * 60 * 60) {
  const self = this;

  if (self.initialized === true) {
    return _bluebird2.default.resolve({
      message: 'QueryEngine already initialized'
    });
  }

  self.cache = null;

  if (enableCache) {
    const defaultSettings = {
      max: cacheSize,
      maxAge: cacheMaxAge
    };
    const lruCache = (0, _lruCache2.default)(defaultSettings);
    const cache = {
      set: function (key, value, maxAge) {
        lruCache.set(key, value, maxAge);
      },
      get: function (key) {
        return lruCache.get(key);
      },
      reset: function () {
        lruCache.reset();
      }
    };
    self.cache = cache;
  }

  return new _bluebird2.default((fulfill, reject) => {
    const succesfullInitializationMsg = { message: 'QueryEngine initialized successfully.' };
    const elasticsearchStatus = _lodash2.default.get(self, 'server.plugins.elasticsearch.status');
    if (elasticsearchStatus && elasticsearchStatus.state === 'green') {
      // already green - fire the _onStatusGreen
      self._onStatusGreen().then(function () {
        fulfill(succesfullInitializationMsg);
      }).catch(reject);
    } else {
      // not ready yet - bind _onStatusGreen to change event so it will fire immediately when it becomes green
      elasticsearchStatus.on('change', function () {
        // eslint-disable-line memoryleaks
        // fire the _onStatusGreen only when elasticsearch status is green
        if (self.server.plugins.elasticsearch.status.state === 'green') {
          self._onStatusGreen().then(function () {
            fulfill(succesfullInitializationMsg);
          }).catch(reject);
        }
      });
    }
  });
};

QueryEngine.prototype.checkESVersion = function () {
  return this.savedObjectsClient.getDocumentFormat(this.savedObjectsAPI.getServerCredentials()).then(format => {
    if (format.version === '5') {
      return 5;
    } else if (format.statusCode === 404 || format.statusCode === 200 && format.version === '6') {
      // if ES6 format or empty elasticsearch
      return 6;
    } else {
      this.log.error('Could not check elasticsearch version');
    }
  }).catch(err => {
    this.log.error(err);
  });
};

QueryEngine.prototype.getSavedObjectById = function (desiredType, id) {
  return this.savedObjectsClient.get(desiredType, id, this.savedObjectsAPI.getServerCredentials());
};

QueryEngine.prototype.loadPredefinedData = function () {
  const self = this;
  return new _bluebird2.default(function (fulfill, reject) {

    const tryToLoad = function () {
      self._isKibiIndexPresent().then(function () {
        self._loadTemplates().then(function () {
          return self._refreshKibiIndex().then(function () {
            fulfill(true);
          });
        }).catch(reject);
      }).catch(function (err) {
        self.log.warn('Could not retrieve Siren Investigate index: ' + err);
        setTimeout(tryToLoad.bind(self), 500);
      });
    };

    return self.checkESVersion().then(esVersion => {
      if (esVersion === 5) {
        self.log.info('Skip template and datasource loading to let migrations run first');
        return fulfill(true);
      } else if (esVersion === 6) {
        return tryToLoad();
      } else {
        self.log.error('Could not check elasticsearch version');
      }
    });
  });
};

QueryEngine.prototype._isKibiIndexPresent = function () {
  return this.cluster.callWithInternalUser('cat.indices', {
    index: this.config.get('kibana.index')
  }).then(kibiIndex => {
    if (!!kibiIndex) {
      this.log.info('Found Siren Investigate index: [' + this.config.get('kibana.index') + ']');
      return true;
    }
    return _bluebird2.default.reject(new Error('Siren Investigate index: [' + this.config.get('kibana.index') + '] does not exist'));
  });
};

QueryEngine.prototype._refreshKibiIndex = function () {
  return this.cluster.callWithInternalUser('indices.refresh', {
    index: this.config.get('kibana.index')
  });
};

QueryEngine.prototype.schema = function (path, options) {
  const opts = {
    method: options.method ? options.method : 'POST',
    uri: options.url + path,
    body: options.data ? options.data : '{}',
    json: true
  };

  if (this.sslCA) {
    opts.ca = this.sslCA;
  }
  return (0, _requestPromise2.default)(opts);
};

QueryEngine.prototype.gremlin = function (credentials, sirenSessionId, options) {
  const gremlinUrl = this.config.get('investigate_core.gremlin_server.url');

  const gremlinOptions = {
    method: options.method | 'GET',
    uri: gremlinUrl + '/graph/queryBatch',
    timeout: 10000 // fixed, as it is not configurable anyway
  };

  if (this.sslCA) {
    gremlinOptions.ca = this.sslCA;
  }

  _lodash2.default.assign(gremlinOptions, options);
  if (gremlinOptions.data) {
    if (sirenSessionId) {
      gremlinOptions.data.sirenSessionId = sirenSessionId;
    }
    gremlinOptions.data.credentials = credentials;
  }
  if (gremlinOptions.json) {
    if (sirenSessionId) {
      gremlinOptions.json.sirenSessionId = sirenSessionId;
    }
    gremlinOptions.json.credentials = credentials;
  }

  return (0, _requestPromise2.default)(gremlinOptions);
};

/**
 *  Issues a ping to the gremlin server.
 */
QueryEngine.prototype.gremlinPing = function (baseGraphAPIUrl) {
  const options = {
    method: 'GET',
    uri: baseGraphAPIUrl + '/ping',
    timeout: 5000
  };

  if (this.sslCA) {
    options.ca = this.sslCA;
  }
  return (0, _requestPromise2.default)(options);
};

QueryEngine.prototype.gremlinReloadOntologyModel = function () {
  const gremlinUrl = this.config.get('investigate_core.gremlin_server.url');

  const options = {
    method: 'GET',
    uri: gremlinUrl + '/schema/reloadOntologyModel',
    timeout: 5000
  };

  if (this.sslCA) {
    options.ca = this.sslCA;
  }

  return (0, _requestPromise2.default)(options);
};

/**
 *  Used to clear index related caches in the gremlin server.
 */
QueryEngine.prototype.gremlinClearCache = function () {
  const gremlinUrl = this.config.get('investigate_core.gremlin_server.url');

  const options = {
    method: 'GET',
    uri: gremlinUrl + '/graph/clearCache',
    timeout: 5000
  };

  if (this.sslCA) {
    options.ca = this.sslCA;
  }

  return (0, _requestPromise2.default)(options);
};

/**
 * Loads default templates.
 *
 * @return {Promise.<*>}
 */
QueryEngine.prototype._loadTemplates = function () {
  const self = this;

  const templatesToLoad = ['kibi-html-angular', 'kibi-json-jade', 'kibi-table-jade', 'kibi-table-handlebars'];

  self.log.info('Loading templates');
  return _bluebird2.default.all(templatesToLoad.map(templateId => {
    return _fs2.default.readFile(_path2.default.join(__dirname, 'templates', templateId + '.json'), function (err, data) {
      if (err) {
        throw err;
      }
      const templateJSON = data.toString();
      const templateData = JSON.parse(templateJSON);
      const templateEs6Id = templateData.title;
      // if we find the script already loaded just skip it, a migration will kick in if necessary.
      return self.getSavedObjectById('template', templateEs6Id).then(es6Template => true).catch(error => {
        if (!(error.statusCode === 404)) {
          return self.log.error(`Could not get template [${templateEs6Id}]: ${error.message}`);
        }

        return _fs2.default.readFile(_path2.default.join(__dirname, 'templates', templateId + '-source' + '.html'), function (sourceError, sourceTemplate) {
          if (!sourceError && sourceTemplate) {
            templateData.templateSource = sourceTemplate.toString();
          }

          self.savedObjectsClient.create('template', templateData, { id: templateEs6Id }, self.savedObjectsAPI.getServerCredentials()).then(() => {
            self.log.info('Template [' + templateEs6Id + '] successfully loaded');
          }).catch(error => {
            self.log.error(`Could not load template [${templateEs6Id}]: ${error.message}`);
          });
        });
      });
    });
  }));
};

QueryEngine.prototype._fetchQueriesFromEs = function () {
  return this.cluster.callWithInternalUser('search', {
    index: this.config.get('kibana.index'),
    type: 'query',
    size: 100
  });
};

QueryEngine.prototype._getDatasourceFromEs = function (datasourceId) {
  return this.cluster.callWithInternalUser('search', {
    index: this.config.get('kibana.index'),
    type: 'datasource',
    q: '_id:' + datasourceId
  }).then(function (result) {
    const datasource = result.hits.hits[0];
    if (!datasource) {
      return _bluebird2.default.reject(new Error(`Datasource with id ${datasourceId} was not found.`));
    }
    datasource._source.id = datasource._id;
    return datasource._source;
  });
};

QueryEngine.prototype.reloadQueries = function () {
  const self = this;
  return self._fetchQueriesFromEs().then(function (resp) {
    const queryDefinitions = [];
    const datasourcesIds = [];
    if (resp.hits && resp.hits.hits && resp.hits.hits.length > 0) {
      self.log.info('Reloading ' + resp.hits.hits.length + ' queries into memory:');
      _lodash2.default.each(resp.hits.hits, function (hit) {
        self.log.info('Reloading [' + hit._id + ']');
        const queryDefinition = {
          id: hit._id,
          label: hit._source.title,
          description: hit._source.description,
          activationQuery: hit._source.activationQuery,
          resultQuery: hit._source.resultQuery,
          datasourceId: hit._source.datasourceId,
          rest_method: hit._source.rest_method,
          rest_path: hit._source.rest_path,
          rest_body: hit._source.rest_body,
          tags: hit._source.tags
        };

        if (datasourcesIds.indexOf(hit._source.datasourceId) === -1) {
          datasourcesIds.push(hit._source.datasourceId);
        }
        // here we are querying the elastic search
        // and rest_params, rest_headers
        // comes back as strings
        try {
          queryDefinition.rest_params = JSON.parse(hit._source.rest_params);
        } catch (e) {
          queryDefinition.rest_params = [];
        }
        try {
          queryDefinition.rest_headers = JSON.parse(hit._source.rest_headers);
        } catch (e) {
          queryDefinition.rest_headers = [];
        }
        try {
          queryDefinition.rest_variables = JSON.parse(hit._source.rest_variables);
        } catch (e) {
          queryDefinition.rest_variables = [];
        }
        try {
          queryDefinition.rest_resp_status_code = parseInt(hit._source.rest_resp_status_code);
        } catch (e) {
          queryDefinition.rest_resp_status_code = 200;
        }
        try {
          queryDefinition.activation_rules = JSON.parse(hit._source.activation_rules);
        } catch (e) {
          queryDefinition.activation_rules = [];
        }

        queryDefinitions.push(queryDefinition);
      });
    }

    if (queryDefinitions.length > 0) {
      return self.cluster.callWithInternalUser('search', {
        index: self.config.get('kibana.index'),
        type: 'datasource',
        size: 100
      }).then(function (datasources) {
        // now as we have all datasources
        // iterate over them and set the clazz
        for (let i = 0; i < datasources.hits.total; i++) {
          const hit = datasources.hits.hits[i];
          if (datasourcesIds.indexOf(hit._id) !== -1) {
            (0, _set_datasource_clazz2.default)(self.server, hit._source);
          }
        }

        self.queries = (0, _lodash2.default)(queryDefinitions).filter(function (queryDef) {
          //filter out queries for which datasources does not exists
          const datasource = _lodash2.default.find(datasources.hits.hits, datasource => datasource._id === queryDef.datasourceId);
          if (datasource) {
            datasource._source.id = datasource._id;
            queryDef.datasource = datasource._source;
            return true;
          }
          self.log.error('Query [' + queryDef.id + '] not loaded because datasource [' + queryDef.datasourceId + '] not found');
          return false;
        }).map(function (queryDef) {
          // now once we have query definitions and datasources load queries
          if (queryDef.datasource.datasourceType === _kibiutils2.default.DatasourceTypes.rest) {
            return new _rest_query2.default(self.server, queryDef, self.cache);
          } else {
            if (['sqlite', 'mysql', 'postgresql', 'sparql_http'].includes(queryDef.datasource.datasourceType)) {
              self.log.warn(`[DEPRECATION] ` + `The ${queryDef.datasource.datasourceType} datasource type has been deprecated, please remove/change query [${queryDef.id}]`);
            } else {
              self.log.error('Unknown datasource type [' + queryDef.datasource.datasourceType + '] - could NOT create query object');
            }
            return false;
          }
        }).value();
      });
    }
  }).catch(function (err) {
    self.log.error('Something is wrong - elasticsearch is not running');
    self.log.error(err);
  });
};

QueryEngine.prototype.clearCache = function () {
  return this._init().then(() => {
    if (this.cache) {
      const promisedReset = _bluebird2.default.method(this.cache.reset);
      return promisedReset().then(() => this.reloadQueries()).then(() => 'Cache cleared, Queries reloaded');
    }
    // here we are reloading queries no matter that cache is enabled or not
    return this.reloadQueries().then(() => 'The cache is disabled, Queries reloaded');
  });
};

/**
 * Returns a ordered list of query objects which:
 * a) do match the URI - this is implemented by executing the ASK query of each of the templates and checking which returns TRUE.
 * b) query label matches the names in queryIds (if provided)
 * Order is given by the priority value.
 */
QueryEngine.prototype._getQueries = function (queryIds, options) {
  if (this.queries.length === 0) {
    return _bluebird2.default.reject(new Error('There are no queries in memory. Create a new query or reload the existing ones from elastic search index'));
  }

  const errors = (0, _lodash2.default)(this.queries).filter(function (query) {
    return query instanceof _error_query2.default;
  }).map(function (err) {
    return err.getErrorMessage();
  }).value();
  if (errors && errors.length !== 0) {
    let msg = '';
    _lodash2.default.each(errors, function (err) {
      msg += err + '\n';
    });
    return _bluebird2.default.reject(new Error(msg));
  }

  const all = !queryIds || queryIds && queryIds.length === 1 && queryIds[0] === 'ALL';

  // if all === false
  // check that all requested queryIds exists and if not reject
  if (!all && queryIds) {
    for (const id of queryIds) {
      let exists = false;

      for (let j = 0; j < this.queries.length; j++) {
        if (id === this.queries[j].id) {
          exists = true;
          break;
        }
      }
      if (!exists) {
        return _bluebird2.default.reject(new Error('The query [' + id + '] was requested by Siren Investigate but not found in memory. ' + 'Possible reason - query datasource was removed. Please check the configuration'));
      }
    }
  }

  const withRightId = _lodash2.default.filter(this.queries, function (query) {
    if (all) {
      return true;
    }
    return _lodash2.default.indexOf(queryIds, query.id) !== -1;
  });

  // here if after filtering we get zero results
  // it means that the requested query is not loaded in memory
  // reject with meaningful error
  if (withRightId.length === 0) {
    return _bluebird2.default.reject(new Error(`Non of the requested queries ${JSON.stringify(queryIds, null, ' ')} were found in memory`));
  }

  return _bluebird2.default.map(withRightId, query => {
    return query.checkIfItIsRelevant(options).then(isQueryRelevant => {
      switch (isQueryRelevant) {
        case _symbols2.default.QUERY_RELEVANT:
          return query;
        case _symbols2.default.QUERY_DEACTIVATED:
          return new _inactivated_query2.default(this.server, query.id, query.config.label);
        case _symbols2.default.SELECTED_DOCUMENT_NEEDED:
          return new _missing_selected_document_query2.default(query.id, query.config.label);
      }
    });
  }).then(function (queryResponses) {
    // order templates as they were ordered in queryIds array
    // but do it only if NOT special case ALL

    if (all) {
      return queryResponses;
    }
    const filteredSortedQueries = [];

    _lodash2.default.each(queryIds, function (id) {
      const found = _lodash2.default.find(queryResponses, 'id', id);
      if (found) {
        filteredSortedQueries.push(found);
      }
    });

    return filteredSortedQueries;
  });
};

QueryEngine.prototype._getQueryDefById = function (queryDefs, queryId) {
  // here grab the corresponding queryDef
  return _lodash2.default.find(queryDefs, function (queryDef) {
    return queryDef.queryId === queryId;
  });
};

// Returns an array with response data from all relevant queries
// Use this method when you need just data and not query html
QueryEngine.prototype.getQueriesData = function (queryDefs, options) {
  const self = this;

  const queryIds = _lodash2.default.map(queryDefs, function (queryDef) {
    return queryDef.queryId;
  });

  return self._init().then(function () {
    return self._getQueries(queryIds, options).then(function (queries) {

      const promises = _lodash2.default.map(queries, function (query) {

        const queryDefinition = self._getQueryDefById(queryDefs, query.id);
        const queryVariableName = queryDefinition ? queryDefinition.queryVariableName : null;
        return query.fetchResults(options, null, queryVariableName);
      });
      return _bluebird2.default.all(promises);
    });
  });
};

QueryEngine.prototype.getQueriesHtml = function (queryDefs, options) {
  const self = this;

  const queryIds = _lodash2.default.map(queryDefs, function (queryDef) {
    return queryDef.queryId;
  });

  return self._init().then(function () {
    return self._getQueries(queryIds, options).then(function (queries) {
      return _bluebird2.default.map(queries, query => {
        const queryDef = self._getQueryDefById(queryDefs, query.id);
        return query.getHtml(queryDef, options);
      });
    });
  });
};

/*
 * Executes given queries and returns only extracted ids
 * Use this method when you need just ids
 *
 * queryDefs is an array of object
 * queryDefs = [
 *     {
 *       queryId: ID,
 *       queryVariableName: VARIABLE_NAME // this property is required only if we want to extract ids
 *    },
 *    ...
 *  ]
 */
QueryEngine.prototype.getIdsFromQueries = function (queryDefs, options) {
  const self = this;

  const queryIds = _lodash2.default.map(queryDefs, function (queryDef) {
    return queryDef.queryId;
  });

  return self._init().then(function () {
    return self._getQueries(queryIds, options).then(function (queries) {
      return _bluebird2.default.map(queries, function (query) {
        const queryDefinition = self._getQueryDefById(queryDefs, query.id);
        const queryVariableName = queryDefinition ? queryDefinition.queryVariableName : null;
        return query.fetchResults(options, true, queryVariableName);
      });
    });
  });
};

module.exports = QueryEngine;
