'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _elasticsearch = require('elasticsearch');

var _elasticsearch2 = _interopRequireDefault(_elasticsearch);

var _joi = require('joi');

var _joi2 = _interopRequireDefault(_joi);

var _joi_to_mapping = require('./_joi_to_mapping');

var _joi_to_mapping2 = _interopRequireDefault(_joi_to_mapping);

var _lodash = require('lodash');

var _requirefrom = require('requirefrom');

var _requirefrom2 = _interopRequireDefault(_requirefrom);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const {
  NotFound
} = _elasticsearch2.default.errors;

const { V6_TYPE } = (0, _requirefrom2.default)('src/server/saved_objects/client/')('saved_objects_client');

/**
 * A model that manages objects having a specific type.
 */
class Model {

  /**
   * Creates a new Model.
   *
   * @param {Server} server - A Server instance.
   * @param {String} type - The Elasticsearch type managed by this model.
   * @param {Joi} schema - A Joi schema describing the type.
   *                       If null, mappings for the object will not be generated.
   * @param {String} title - Optional type title.
   */
  constructor(server, type, schema, title) {
    this._server = server;
    this._type = type;
    this._title = title ? title : this._type;
    this._schema = schema;

    // This is for when we create a class just to grab its schema
    if (server) {
      this._plugin = server.plugins.saved_objects_api;
      this._config = server.config();

      this._cluster = server.plugins.elasticsearch.getCluster('admin');
    }
  }

  /**
   * Returns the api version of the ES Cluster.
   * e.g: "5.x"
   * @return {String}
   */
  get apiVersion() {
    return this._cluster._config.apiVersion;
  }

  /**
   * Returns the schema of the type managed by this model.
   *
   * @return {Joi}
   */
  get schema() {
    return this._schema;
  }

  /**
   * Returns the type managed by this model.
   *
   * @return {String}
   */
  get type() {
    return this._type;
  }

  /**
   * Returns the title of the type managed by this model.
   *
   * @return {String}
   */
  get title() {
    return this._title;
  }

  /**
   * Wraps an Elasticsearch error and throws it.
   *
   * @param {Error} error - An Elasticsearch error.
   * @private
   */
  _wrapError(error) {
    throw error;
  }

  /**
   * Prepares an object body before sending to the backend.
   *
   * @param {Object} body - An object.
   */
  _prepare(body) {}

  /**
   * Sets credentials extracted from the specified HAPI @request, if any.
   * @private
   */
  _setCredentials(parameters, request) {
    const headerPath = 'headers.authorization';
    const authorizationHeader = (0, _lodash.get)(request, headerPath);
    if (authorizationHeader) {
      (0, _lodash.set)(parameters, headerPath, authorizationHeader);
    }
  }

  /**
   * Creates the mappings for the type managed by this model.
   *
   * @param {Object} request - An optional HAPI request or an object
   *                            containing Elasticsearch client parameters.
   */
  async createMappings(request) {
    if (await this.hasMappings(request)) {
      return;
    }
    const body = {};
    const parameters = {
      index: this._config.get('kibana.index'),
      body: body
    };
    parameters.type = V6_TYPE;
    body.dynamic = 'true';
    body.properties = {
      type: {
        type: 'keyword'
      },
      updated_at: {
        type: 'date'
      }
    };
    body.properties[this._type] = {
      dynamic: true,
      properties: (0, _joi_to_mapping2.default)(this.schema)
    };

    this._setCredentials(parameters, request);
    return await this._cluster.callWithRequest({}, 'indices.putMapping', parameters);
  }

  /**
   * Checks if the mappings for the type have been defined.
   *
   * @param {Object} request - An optional HAPI request or an object
   *                            containing Elasticsearch client parameters.
   */
  async hasMappings(request) {
    if (!this.schema) {
      return true;
    }
    const parameters = {
      index: this._config.get('kibana.index')
    };
    parameters.type = V6_TYPE;
    if (request.type !== V6_TYPE) {
      request.type = V6_TYPE;
      if (request.id) {
        request.id = request.id.startsWith(this._type) ? request.id : `${this._type}:${request.id}`;
      }
      const body = request.body;
      request.body = {};
      request.body[this._type] = body;
      request.body.type = this.type;
    }
    this._setCredentials(parameters, request);

    // NOTE:
    // There is a difference between elasticsearch 5.4.x and 5.5.x
    // when there is no mappings
    // 5.4.x resolve with an empty object
    // 5.5.x rejects with a proper Error object
    try {
      const mappings = await this._cluster.callWithRequest({}, 'indices.getMapping', parameters);
      if (parameters.type === V6_TYPE) {
        return (0, _lodash.get)(mappings[parameters.index], `mappings.${parameters.type}.properties.${this._type}`);
      } else {
        return (0, _lodash.get)(mappings[parameters.index], `mappings.${parameters.type}`);
      }
    } catch (error) {
      // throw if an Error is different than NotFound
      if (!(error.statusCode === 404 && error.displayName === 'NotFound')) {
        throw error;
      }
      return false;
    }
  }

  /**
   * Creates a new object instance.
   *
   * Middlewares can:
   *
   * - validate the parameters by implementing createRequest and updateRequest
   * - return custom ES client parameters from createRequest and updateRequest;
   *   parameters are merged.
   * - alter the return value  by implementing createResponse.
   *
   * @param {String} id - The object id.
   * @param {Object} body - The object body.
   * @param {Object} request - An optional HAPI request.
   */
  async create(id, body, request) {
    try {
      let requestMiddlewareMethod = 'createRequest';
      const responseMiddlewareMethod = 'createResponse';

      let response = await this._cluster.callWithInternalUser('get', {
        index: this._config.get('kibana.index'),
        type: V6_TYPE,
        id: id,
        ignore: [404]
      });
      if (response && response.found) {
        requestMiddlewareMethod = 'updateRequest';
      }

      const parameters = {
        id: id,
        index: this._config.get('kibana.index'),
        type: V6_TYPE,
        body,
        refresh: true
      };
      this._setCredentials(parameters, request);

      for (const middleware of this._plugin.getMiddlewares()) {
        (0, _lodash.merge)(parameters, (await middleware[requestMiddlewareMethod](this, id, body, request)));
      }

      this._prepare(body);

      await this.createMappings(parameters);

      response = await this._cluster.callWithRequest({}, 'create', parameters);
      for (const middleware of this._plugin.getMiddlewares()) {
        await middleware[responseMiddlewareMethod](this, id, body, request, response);
      }
      return response;
    } catch (error) {
      this._wrapError(error);
    }
  }

  /**
   * Updates an existing object.
   *
   * Middlewares can:
   *
   * - validate the parameters by implementing updateRequest
   * - return custom ES client parameters from updateRequest; parameters are merged.
   * - alter the return value  by implementing createResponse.
   *
   * @param {String} id - The object id.
   * @param {Object} body - The object body.
   * @param {Object} request - An optional HAPI request.
   */
  async update(id, body, request) {
    try {
      let requestMiddlewareMethod = 'updateRequest';
      let responseMiddlewareMethod = 'updateResponse';
      if (id) {
        const parameters = {
          index: this._config.get('kibana.index'),
          id: id,
          type: V6_TYPE,
          ignore: [404]
        };
        const response = await this._cluster.callWithInternalUser('get', parameters);
        if (!response.found) {
          requestMiddlewareMethod = 'createRequest';
          responseMiddlewareMethod = 'createResponse';
        }
      } else {
        requestMiddlewareMethod = 'createRequest';
        responseMiddlewareMethod = 'createResponse';
      }
      const parameters = {
        id: id,
        index: this._config.get('kibana.index'),
        type: V6_TYPE,
        body: body,
        refresh: true
      };
      this._setCredentials(parameters, request);

      for (const middleware of this._plugin.getMiddlewares()) {
        (0, _lodash.merge)(parameters, (await middleware[requestMiddlewareMethod](this, id, body, request)));
      }

      this._prepare(body);

      await this.createMappings(parameters);

      const response = await this._cluster.callWithRequest({}, 'index', parameters);
      for (const middleware of this._plugin.getMiddlewares()) {
        await middleware[responseMiddlewareMethod](this, id, body, request, response);
      }
      return response;
    } catch (error) {
      this._wrapError(error);
    }
  }

  /**
   * Partially updates an existing object.
   *
   * Middlewares can:
   *
   * - validate the parameters by implementing patchRequest
   * - return custom ES client parameters from patchRequest; parameters are merged.
   * - alter the return value  by implementing patchResponse.
   *
   * @param {String} id - The object id.
   * @param {Object} fields - The changed fields.
   * @param {Object} request - An optional HAPI request.
   */
  async patch(id, fields, request) {
    try {
      const parameters = {
        id: id,
        index: this._config.get('kibana.index'),
        type: V6_TYPE,
        body: {
          doc: {
            type: this._type,
            [this._type]: fields
          }
        },
        refresh: true
      };

      this._setCredentials(parameters, request);

      for (const middleware of this._plugin.getMiddlewares()) {
        (0, _lodash.merge)(parameters, (await middleware.patchRequest(this, id, fields, request)));
      }

      const response = await this._cluster.callWithRequest({}, 'update', parameters);
      for (const middleware of this._plugin.getMiddlewares()) {
        await middleware.patchResponse(this, id, fields, request, response);
      }
      return response;
    } catch (error) {
      this._wrapError(error);
    }
  }

  /**
   * Searches objects of the type managed by this model.
   *
   * Middlewares can:
   *
   * - validate the parameters by implementing patchRequest
   * - return custom ES client parameters from patchRequest; parameters are merged.
   * - alter the return value  by implementing patchResponse.
   *
   * @param {Number} size - The number of results to return. If not set, returns all objects matching the search.
   * @param {String} search - An optional search string or query body.
   * @param {Object} request - Optional HAPI request.
   * @param {Array} exclude - An optional list of fields to exclude.
   * @return {Array} A list of objects of the specified type.
   * @throws {NotFound} if the object does not exist.
   */
  async search(size, search, request, exclude, ignoreBroken = false) {
    const excludedIndices = search.excludedIndices || [];

    try {
      const commonParameters = {};
      this._setCredentials(commonParameters, request);

      let body;
      if (search) {
        if ((0, _lodash.isString)(search)) {
          body = {
            query: {
              simple_query_string: {
                query: `${search}*`,
                fields: ['title^3', 'description'],
                default_operator: 'AND'
              }
            }
          };
        } else {
          body = search.createFindQuery((0, _joi_to_mapping2.default)(this.schema));
          search = body;
        }
      } else {
        body = {
          query: {
            match_all: {}
          }
        };
      }

      for (const middleware of this._plugin.getMiddlewares()) {
        (0, _lodash.merge)(commonParameters, (await middleware.searchRequest(this, size, search, request)));
      }

      let parameters = {
        index: this._config.get('kibana.index'),
        body: body
      };
      (0, _lodash.merge)(parameters, commonParameters);

      if (exclude && exclude.length > 0) {
        parameters.body._source = {
          excludes: []
        };
        for (const ele of exclude) {
          if (!ele.startsWith(this.type)) {
            parameters.body._source.excludes.push(`${this.type}.${ele}`);
          } else {
            parameters.body._source.excludes.push(ele);
          }
        }
      }

      if (size === 0) {
        parameters.size = 0;
      } else {
        parameters.size = 100;
        parameters.scroll = '1m';
      }

      let response = await this._cluster.callWithRequest({}, 'search', parameters);
      let scrollId = response._scroll_id;

      if (scrollId) {
        const hits = [];
        while (true) {
          hits.push(...response.hits.hits);
          if (hits.length === response.hits.total) {
            break;
          }
          parameters = (0, _lodash.merge)({
            scroll: '1m',
            scrollId
          }, commonParameters);
          response = await this._cluster.callWithRequest({}, 'scroll', parameters);
          scrollId = response._scroll_id;
        }

        parameters = (0, _lodash.merge)({
          scrollId
        }, commonParameters);

        try {
          await this._cluster.callWithRequest({}, 'clearScroll', parameters);
        } catch (error) {
          // ignore errors on clearScroll
        }

        response = {
          hits: {
            hits: hits,
            total: hits.length
          }
        };
      }

      for (const middleware of this._plugin.getMiddlewares()) {
        await middleware.searchResponse(this, size, search, request, response);
      }

      response = await this.filterSearchResponse(request, response, excludedIndices, ignoreBroken);
      return response;
    } catch (error) {
      this._wrapError(error);
    }
  }

  /**
   * Filters the response of the search function.
   * This has to be implemented by the subclass to add specific behavior.
   *
   * @param {Object} request - Optional HAPI request.
   * @param {Array} searchResponse - The result of the search function.
   * @param {Array} excludedIndices - A list of excluded indices.
   * @return {Array} A list of objects of the specified type.
   */
  async filterSearchResponse(request, searchResponse, excludedIndices) {
    return searchResponse;
  }

  /**
   * Counts objects of the type managed by this model.
   * @param {Object} request - Optional HAPI request.
   * @param {String} search - An optional search string or query body.
   * @return {Number} The number of objects of the type managed by this model.
   */
  async count(search, request) {
    const response = await this.search(0, search, request, null);
    return response.hits.total;
  }

  /**
   * Returns the object with the specified id.
   *
   * Arguments and response can be modified or validated by middlewares.
   *
   * @param {String} id - An id.
   * @param {Object} request - Optional HAPI request.
   * @return {Object} The object instance having the specified id.
   * @throws {NotFound} if the object does not exist.
   */
  async get(id, request, options) {
    try {
      const parameters = {
        index: this._config.get('kibana.index'),
        type: V6_TYPE,
        id: id
      };
      this._setCredentials(parameters, request);

      for (const middleware of this._plugin.getMiddlewares()) {
        (0, _lodash.merge)(parameters, (await middleware.getRequest(this, id, request)));
      }

      const response = await this._cluster.callWithRequest({}, 'get', parameters, options);
      for (const middleware of this._plugin.getMiddlewares()) {
        await middleware.getResponse(this, id, request, response);
      }
      return response;
    } catch (error) {
      if (error.statusCode === 404) {
        throw new NotFound(`${id} does not exist.`, error);
      }
      this._wrapError(error);
    }
  }

  /**
   * Deletes the object with the specified id.
   *
   * Arguments can be modified or validated by middlewares.
   *
   * @param {String} id - An id.
   * @param {Function} createIdQuery - Finds a document by either its v5 or v6 format.
   * @param {Object} request - Optional HAPI request.
   * @throws {NotFound} if the object does not exist.
   */
  async delete(id, createIdQuery, request) {
    const parameters = {
      body: createIdQuery({ type: this._type, id }),
      refresh: 'wait_for'
    };
    this._setCredentials(parameters, request);
    for (const middleware of this._plugin.getMiddlewares()) {
      (0, _lodash.merge)(parameters, (await middleware.deleteRequest(this, id, request)));
    }

    await this._cluster.callWithRequest({}, 'deleteByQuery', _extends({}, parameters, {
      index: this._config.get('kibana.index')
    }));
    for (const middleware of this._plugin.getMiddlewares()) {
      await middleware.deleteResponse(this, id, request);
    }
  }

  // kibi: added function
  /**
   * Gets the document format by reading the config object (if present).
   *
   * @param {Object} request - HAPI request.
   * @throws {NotFound} if the object does not exist.
   */
  async getDocumentFormat(request) {
    try {
      const es6doc = await this.get('config:siren', request);
      return '6';
    } catch (error) {
      // if not found try the es5 format
      if (error.status === 404) {
        try {
          const parameters = {
            index: this._config.get('kibana.index'),
            type: 'config',
            id: 'siren'
          };
          this._setCredentials(parameters, request);

          const response = await this._cluster.callWithRequest({}, 'get', parameters, {});
          return '5';
        } catch (error) {
          if (error.statusCode === 404) {
            throw new NotFound(`config object does not exist.`, error);
          }
          this._wrapError(error);
        }
      }
    }
  }
  // kibi: end
}
exports.default = Model;
module.exports = exports['default'];
