import _ from 'lodash';
import Node from './node';
import TreeType from './tree_type';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { KibiSequentialJoinVisHelperFactory } from 'ui/kibi/helpers/kibi_sequential_join_vis_helper';
import { QueryBuilderFactory } from 'ui/kibi/helpers/query_builder';

export function SirenAutoJoinHelperProvider(Private, Promise, es, kibiState, kbnUrl, indexPatterns, getAppState, createNotifier) {

  const sirenSequentialJoinVisHelper = Private(KibiSequentialJoinVisHelperFactory);
  const queryBuilder = Private(QueryBuilderFactory);
  const notify = createNotifier({
    location: 'Auto Joins Helper'
  });

  function omitDeep(obj, omitKey) {
    delete obj[omitKey];

    _.each(obj, function (val, key) {
      if (val && typeof (val) === 'object') {
        obj[key] = omitDeep(val, omitKey);
      }
    });

    return obj;
  }

  class SirenAutoJoinHelper {

    createTreeRoot() {
      return new Node({
        id: 'root',
        showChildren: true,
        parent: null
      });
    }

    createFirstLevelNodes(tree, buttons, btnCountEnabled) {
      _.each(buttons, button => {
        if (btnCountEnabled) {
          button.showSpinner = true;
        }
        const node = new Node({
          type: button.type === EntityType.SAVED_SEARCH ? TreeType.BUTTON : TreeType.VIRTUAL_BUTTON,
          id: button.id,
          label: button.label,
          showChildren: false,
          visible: true,
          useAltNodes: false,
          button: button
        });
        tree.addNode(node);
      });

      tree.nodes.sort(function (a, b) {
        if (a.type === TreeType.BUTTON && b.type === TreeType.VIRTUAL_BUTTON) {
          return -1;
        } else if (b.type === TreeType.BUTTON && a.type === TreeType.VIRTUAL_BUTTON) {
          return 1;
        }
        return 0;
      });
    }

    _constructSubButton(parentButton, dashboard, relation, label, labelParts) {
      return {
        id: relation.id + '-ip-' + dashboard.title,
        indexRelationId: parentButton.indexRelationId,
        label: label,
        labelParts: labelParts,
        sourceDashboardId: parentButton.sourceDashboardId,
        sourceField: parentButton.sourceField,
        sourceIndexPatternId: parentButton.sourceIndexPatternId,
        targetDashboardId: dashboard.id,
        targetField: relation.range.field,
        targetIndexPatternId: relation.range.indexPatternId,
        type: EntityType.SAVED_SEARCH,
        targetCount: ''
      };
    };

    addNodesToTreeLightLayout(tree, relations, compatibleSavedSearchesMap, compatibleDashboardsMap, btnCountEnabled, visibility) {
      for (let i = tree.nodes.length - 1; i >= 0; i--) {
        const node = tree.nodes[i];

        if (node.type === TreeType.VIRTUAL_BUTTON) {
          const relationsByDomain = _.filter(relations, rel => rel.domain.indexPatternId === node.button.targetIndexPatternId);
          _.each(relationsByDomain, relByDomain => {
            const key = relByDomain.directLabel;
            if (!visibility || !visibility[node.id] || !visibility[node.id].relation
              || !visibility[node.id].relation[key] || visibility[node.id].relation[key].toggle !== false) {
              if (relByDomain.range.indexPatternId !== node.button.sourceIndexPatternId) {
                // filter the savedSearch with the same indexPattern
                const compatibleSavedSearches = compatibleSavedSearchesMap[relByDomain.range.indexPatternId];
                _.each(compatibleSavedSearches, compatibleSavedSearch => {
                  const compatibleDashboards = compatibleDashboardsMap[compatibleSavedSearch.id];
                  _.each(compatibleDashboards, compatibleDashboard => {
                    const subButton = this._constructSubButton(
                      node.button,
                      compatibleDashboard,
                      relByDomain,
                      relByDomain.directLabel + ' ' + compatibleDashboard.title + ' ({0})',
                      [relByDomain.directLabel, compatibleDashboard.title, '({0})']
                    );
                    if (btnCountEnabled) {
                      subButton.showSpinner = true;
                    }
                    if (!visibility || !visibility[node.id] || !visibility[node.id].relation
                      || !visibility[node.id].relation[key] || !visibility[node.id].relation[key].dashboard
                      || visibility[node.id].relation[key].dashboard[compatibleDashboard.title] !== false) {
                      const buttonNode = new Node({
                        type: TreeType.BUTTON,
                        id: subButton.id,
                        visible: false,
                        showChildren: false,
                        useAltNodes: false,
                        button: subButton
                      });

                      node.addNode(buttonNode);
                    }
                  });
                });
              }
            }

          });
          // now if there are buttons with no dashboard button prune them
          // can happen if user has no access to destination dashboard
          // or destination dashboard was removed
          if (node.nodes.length === 0) {
            tree.nodes.splice(i, 1);
          }
        } else {
          if (btnCountEnabled) {
            node.button.showSpinner = true;
          }
        }
      }
    }

    _addClickHandlerToNode(node, clickHandlerFunction) {
      if (node.type === TreeType.BUTTON && node.button) {
        node.button.click = function () {
          clickHandlerFunction(node.button);
        };
      } else {
        _.each(node.nodes, n => this._addClickHandlerToNode(n, clickHandlerFunction));
        _.each(node.altNodes, n => this._addClickHandlerToNode(n, clickHandlerFunction));
      }
    }

    addButtonClickHandlerToTree(tree, clickHandlerFunction) {
      this._addClickHandlerToNode(tree, clickHandlerFunction);
    }

    addNodesToTreeNormalLayout(tree, relations, compatibleSavedSearchesMap, compatibleDashboardsMap, btnCountEnabled, visibility) {
      const currentDashboardId = kibiState.getCurrentDashboardId();

      for (let i = tree.nodes.length - 1; i >= 0; i--) {
        const node = tree.nodes[i];

        if (node.type === TreeType.VIRTUAL_BUTTON) {
          const relationsByDomain = _.filter(relations, rel => rel.domain.indexPatternId === node.button.targetIndexPatternId);
          _.each(relationsByDomain, relByDomain => {
            const key = relByDomain.directLabel;
            const relNodeId = 'tree-relation-' + key;

            let relNode = node.findNode(relNodeId);

            if (!visibility || !visibility[node.id] || !visibility[node.id].relation
                || !visibility[node.id].relation[key] || visibility[node.id].relation[key].toggle !== false) {
              // filter the savedSearch with the same indexPattern
              const compatibleSavedSearches = compatibleSavedSearchesMap[relByDomain.range.indexPatternId];
              _.each(compatibleSavedSearches, compatibleSavedSearch => {
                const compatibleDashboards = compatibleDashboardsMap[compatibleSavedSearch.id];
                _.each(compatibleDashboards, compatibleDashboard => {
                  if (compatibleDashboard.id !== currentDashboardId) {
                    const subButton = this._constructSubButton(
                      node.button,
                      compatibleDashboard,
                      relByDomain,
                      compatibleDashboard.title + ' ({0})',
                      [compatibleDashboard.title, '({0})']
                    );
                    if (btnCountEnabled) {
                      subButton.showSpinner = true;
                    }

                    if (!relNode) {
                      relNode = new Node({
                        type: TreeType.RELATION,
                        id: relNodeId,
                        label: key,
                        showChildren: false,
                        visible: true,
                        useAltNodes: false,
                      });

                      node.addNode(relNode);
                    }

                    if (!visibility || !visibility[node.id] || !visibility[node.id].relation
                     || !visibility[node.id].relation[key] || !visibility[node.id].relation[key].dashboard
                     || visibility[node.id].relation[key].dashboard[compatibleDashboard.title] !== false) {
                      const buttonNode = new Node({
                        type: TreeType.BUTTON,
                        id: subButton.id,
                        label: subButton.label,
                        visible: false,
                        showChildren: false,
                        useAltNodes: false,
                        button: subButton
                      });

                      relNode.addNode(buttonNode);
                    }
                  }
                });
              });
            }
          });
          // now if there are relations with no children prune them
          // can happen if user has no access to destination dashboard
          // or destination dashboard was removed

          // if relation node has no children prune it
          for (let j = node.nodes.length - 1; j >= 0; j--) {
            const relNode = node.nodes[j];
            if (relNode.nodes.length === 0) {
              node.nodes.splice(j, 1);
            }
          }

          // if there is no relation nodes left prune the node
          if (node.nodes.length === 0) {
            tree.nodes.splice(i, 1);
          }

        } else {
          if (btnCountEnabled) {
            node.button.showSpinner = true;
          }
        }
      };
    }

    getCardinalityCount(indexPatternId, field, dashboardId = undefined) {
      const fakeButton = { sourceIndexPatternId: indexPatternId, sourceField: field };
      return this._getCardinalityQuery(fakeButton, dashboardId)
        .then(query => es.search(query))
        .then(res => res.aggregations.distinct_field.value);
    }

    /**
     *  Compose the cardinality query for EID buttons using the current dashboard filters.
     */
    _getCardinalityQuery(button, dashboardId = undefined) {
      if (!dashboardId) {
        dashboardId = kibiState.getCurrentDashboardId();
      }

      return kibiState.getState(dashboardId)
        .then(({ index, filters, queries, time }) => {
          // Removes the $state object from filters if present, as it will break the count query.
          const cleanedFilters = omitDeep(filters, '$state');
          const queryDef = queryBuilder(cleanedFilters, queries, time);

          queryDef._source = false;
          queryDef.size = 0;
          queryDef.aggregations = { distinct_field : { cardinality : { field : button.sourceField } } };

          return indexPatterns.get(button.sourceIndexPatternId)
            .then(indexPattern => {
              return {
                index: indexPattern.title,
                body: queryDef,
                ignore_unavailable: true // has to be here to correctly handle index exclusions in joins
              };
            });
        });
    };

    updateTreeCardinalityCounts(tree) {
      const promises = [];
      _.each(tree.nodes, node => {
        if (node.type === TreeType.VIRTUAL_BUTTON) {
          node.button.targetCount = ''; // show ...
          const p = this._getCardinalityQuery(node.button)
            .then(cardinalityQuery => {
              node.button.targetCount = undefined; // turn on spinner
              return es.search(cardinalityQuery);
            })
            .then(esResult => node.button.targetCount = esResult.aggregations.distinct_field.value)
            .catch(err => {
              node.button.targetCount = ''; // show ...
              node.button.warning = 'Got an unexpected error while computing cardinality counts';
              notify.error(err);
            });

          promises.push(p);
        }
      });
      return Promise.all(promises)
        .then(() => tree);
    };

    _addVirtualButtonIfVisible(node, buttons, layout) {
      // if normal layout
      if (layout === 'normal') {
        if (!node.useAltNodes) {
          _.each(node.nodes, relNode => {
            if (relNode.visible) {
              _.each(relNode.nodes, buttonNode => {
                if (buttonNode.visible) {
                  buttons.push(buttonNode.button);
                }
              });
            }
          });
        }
        if (node.useAltNodes) {
          _.each(node.altNodes, dashNode => {
            if (dashNode.visible) {
              _.each(dashNode.nodes, buttonNode => {
                if (buttonNode.visible) {
                  buttons.push(buttonNode.button);
                }
              });
            }
          });
        }
      } else if (layout === 'light') {
        _.each(node.nodes, dashRelNode => {
          if (dashRelNode.visible) {
            buttons.push(dashRelNode.button);
          }
        });
      }
    }

    getVisibleVirtualEntitySubButtons(tree, layout) {
      const buttons = [];
      _.each(tree.nodes, node => {
        if (node.type === TreeType.VIRTUAL_BUTTON) {
          this._addVirtualButtonIfVisible(node, buttons, layout);
        }
      });
      return buttons;
    }

    getAllVisibleButtons(tree, layout) {
      const buttons = [];
      _.each(tree.nodes, node => {
        if (node.type === TreeType.BUTTON) {
          // always visible on the first level
          buttons.push(node.button);
        } else if (node.type === TreeType.VIRTUAL_BUTTON) {
          this._addVirtualButtonIfVisible(node, buttons, layout);
        } else {
          throw 'Wrong type at first level of the tree';
        }
      });
      return buttons;
    };

    switchDashboard(button) {
      return kibiState.saveAppState().then(() => {
        kibiState.addFilter(button.targetDashboardId, button.joinSeqFilter);
        kibiState.save(false, true);
        const currentDashboardId = kibiState.getCurrentDashboardId();
        // kibi: if it is self relation add filters to appState
        if (currentDashboardId === button.targetDashboardId) {
          const appState = getAppState();
          appState.filters.push(button.joinSeqFilter);
          appState.save();
        } else {
          kbnUrl.change('/dashboard/{{id}}', { id: button.targetDashboardId });
        }
      });
    }

    updateFilterLabel(currentDashboardTitle, button, count) {
      if (button.joinSeqFilter) {
        let alias = 'coming from ($COUNT) docs on: $DASHBOARD';
        button.joinSeqFilter.meta.alias_tmpl = alias;

        if (alias.indexOf('$DASHBOARD') !== -1) {
          alias = alias.replace(/\$DASHBOARD/g, currentDashboardTitle);
        }
        if (alias.indexOf('$COUNT') !== -1) {
          alias = alias.replace(/\$COUNT/g, count);
        }
        button.joinSeqFilter.meta.alias = alias;
      }
    }

    constructButtonDefinitions(relations, entities, compatibleSavedSearchesMap, compatibleDashboardsMap, visibility) {
      const buttons = [];
      _.each(relations, rel => {
        if (rel.domain.type === EntityType.SAVED_SEARCH) {
          const button = {
            type: rel.range.type,
            indexRelationId: rel.id,
            domainIndexPattern: rel.domain.indexPatternId,
            sourceDashboardId: null,
            targetDashboardId: null,
            status: 'default'
          };

          if (button.type === EntityType.VIRTUAL_ENTITY) {
            const id = rel.id + '-ve-' + rel.range.id;
            if (!visibility[id] || visibility[id].button !== false) {
              const virtualEntity = _.find(entities, 'id', rel.range.id);
              button.id = id;
              button.label = rel.directLabel + ' ({0} ' + virtualEntity.label + ')';
              button.labelParts = [rel.directLabel, '{0}', virtualEntity.label];
              buttons.push(button);
            }
          } else if (button.type === EntityType.SAVED_SEARCH) {
            const compatibleSavedSearches = compatibleSavedSearchesMap[rel.range.indexPatternId];
            _.each(compatibleSavedSearches, compatibleSavedSearch => {
              const compatibleDashboards = compatibleDashboardsMap[compatibleSavedSearch.id];
              _.each(compatibleDashboards, compatibleDashboard => {
                const id = rel.id + '-ip-' + compatibleDashboard.title;
                if (!visibility[id] || visibility[id].button !== false) {
                  const clonedButton = _.clone(button);
                  clonedButton.targetDashboardId = compatibleDashboard.id;
                  clonedButton.id = id;
                  clonedButton.label = rel.directLabel + ' ({0} ' + compatibleDashboard.title + ')';
                  clonedButton.labelParts = [rel.directLabel, '{0}', compatibleDashboard.title];
                  buttons.push(clonedButton);
                }
              });
            });
          }
        }
      });

      return buttons;
    }

  }

  return new SirenAutoJoinHelper();
}
