import { BaseModalProvider } from './base_modal';

import progressTemplate from './progress_modal.html';
import './progress_modal.less';

import { promiseMapSeries } from 'ui/kibi/utils/promise';
import _ from 'lodash';


/** @class Progress
 * @property {Number}   max               Total number of progress steps
 * @property {Number}   value             Current progress step
 * @property {String}   text              Progress text
 * @property {Boolean}  canceled          Whether execution has been canceled
 * @property {Boolean}  canceledPromise   Rejects with undefined on cancelation
 */

/** @method Progress#notifyStart
 *
 * @description
 * Advances the progress bar and sets the specified text for display.
 * The return value is a negated {@see Progress#canceled}, provided for convenience.
 *
 * @param {String}      text          New progress text
 * @param {Number}      [steps=1]     Incremented steps
 * @return {Boolean}                  Whether the progress can continue
 */

/** @method Progress#updateMax
 *
 * @description
 * In case the overall progresses max steps changes in progress,
 * this method can be called to refresh it.
 *
 * The update will take place at the end of current operation.
 */

/** @function progressMap
 *
 * @description
 * Takes an array of values or promises and *sequentially*
 * maps them into promises, while detailing progress with
 * a modal progress bar.
 *
 * @param {Array} arr
 *    List of values to be mapped and tracked for progress
 *
 * @param {Object} opts
 *    Keyword arguments
 *
 * @param {Function} opts.title
 *    Title for the progress bar modal
 *
 * @param {Function} opts.valueMap
 *    Map from input value to output value. Supplied parameters
 *    are the input value, its index, and the {@link Progress}
 *    context.
 *
 * @param {Function} opts.stepMap
 *    Map from input value to the expected progress step:
 *
 *     - The progress text to show, progress is incremented automatically as a unit.
 *     - The expected number of progresses, to be incremented manually using
 *       {@see Progress#notifyStart}.
 *
 * @param {Function} [opts.className]
 *    Optional class to be added to the progress modal
 *
 * @param {String} [opts.textTemplate]
 *    Optional template to be used for the progress text. Defaults to '{{progress.text}}'.
 *
 * @param {Boolean} [opts.showProgress=true]
 *    Whether the progress bar will be visible or not.
 *
 *    This is mainly available as a way to disable the progress bar on a parametr,
 *    without having to redesign the input promise chain.
 *
 * @param {Object} [opts.nestedIn]
 *    The progress map can be nested inside another specifed progress map.
 * @param {Progress} [opts.nestedIn.prefix=""]
 *    Text prefixed to every progress step.
 * @param {Progress} [opts.nestedIn.progress]
 *    Super-Progress this progress will be nested in.
 * @param {Number} [opts.nestedIn.steps]
 *    The size of this progress bar in the host progress bar.
 *
 * @return {Promise[]}
 *    The mapped values, according to the valueMap.
 */

export function ProgressMapProvider(Private) {
  const baseModal = Private(BaseModalProvider);

  function makeProgressModal(progress, opts) {
    const {
      textTemplate = '{{ progress.text }}',
      title = 'In Progress...',
      className
    } = opts;

    const startTime = Date.now();
    const template = progressTemplate.replace('[[textTemplate]]', textTemplate);

    const modal = baseModal(template, {
      className,
      progress,
      title,

      eta: _.throttle(function eta() {
        const { current, max } = this.progress;

        if (current <= 0) { return '--:--:--'; }

        let time = (max - current) * (Date.now() - startTime) / current;
        time = Math.floor(time * 1e-3);

        let ss = time % 60;
        time = Math.round((time - ss) / 60);
        ss = _.padLeft('' + ss, 2, '0');

        let mm = time % 60;
        time = Math.round((time - mm) / 60);
        mm = _.padLeft('' + mm, 2, '0');

        return `${time}:${mm}:${ss}`;
      }, 400),

      percentCompletion() {
        return Math.floor(100 * this.progress.current / this.progress.max);
      }
    });

    modal.scope.onCancel = function () {
      // Overriding cancel - modal shall not hide until
      // current operation finishes
      this.progress.cancel();
    };

    return modal;
  }

  function makeProgress(arr, opts) {
    const { valueMap, countMap, showProgress = true } = opts;

    let resolveCanceled;
    const canceledPromise = new Promise(function (resolve) {
      resolveCanceled = resolve;
    });

    let max = _.sum(arr, countMap);
    let current = 0;
    let startedSteps = 0;
    let text = '';
    let updateMax = false;
    let canceled = false;

    const progress = {
      get max() { return max; },
      get current() { return current; },
      get text() { return text; },
      get canceled() { return canceled; },
      get canceledPromise() { return canceledPromise; },

      notifyStart(text_, steps = 1) {
        current += startedSteps;
        startedSteps = steps;
        text = text_;

        return !this.canceled;
      },

      updateMax() { updateMax = true; },

      onValue(val, idx) {
        if (!updateMax) { return; }
        updateMax = false;

        max = Math.min(current + _(arr).slice(idx).sum(countMap), max);
      },

      cancel() {
        canceled = true;
        resolveCanceled();
        text = 'Canceling...';
      }
    };

    const modal = showProgress && makeProgressModal(progress, opts);

    progress.waitForReady = Promise.resolve(modal &&
      (modal.show(), modal.waitForVisible));

    progress.close = () => modal && modal.scope.onConfirm();

    return progress;
  }

  function makeSubProgress(arr, opts) {
    const { progress, prefix, steps } = opts.nestedIn;
    opts = _.defaults({ showProgress: false }, opts);

    const subProgress = makeProgress(arr, opts);
    let subStepSize = steps / subProgress.max;

    _.forEach(['canceled', 'canceledPromise'], property => {
      Object.defineProperty(subProgress, property, {
        get: function () { return progress[property]; }
      });
    });

    subProgress.notifyStart = _.wrap(subProgress.notifyStart,
      function (notifyStart, text, subSteps) {
        notifyStart.call(this, prefix + text, subSteps);
        return progress.notifyStart(prefix + text, subSteps && (subSteps * subStepSize));
      });
    subProgress.onValue = _.wrap(subProgress.onValue, function (onValue, val, idx) {
      onValue.call(this, val, idx);
      subStepSize = steps / this.max;
    });
    subProgress.cancel = function (cancel) { progress.cancel(); };

    return subProgress;
  }

  return function progressMap(arr, opts) {
    const valueMap = _.iteratee(opts.valueMap);
    const stepMap = _.iteratee(opts.stepMap);

    function countMap(val, idx) {
      val = stepMap(val, idx);
      return _.isString(val) ? 1 : val;
    }

    opts = _.defaults({ valueMap, stepMap, countMap }, opts);

    const progress = opts.nestedIn
      ? makeSubProgress(arr, opts)
      : makeProgress(arr, opts);


    return Promise.resolve(progress.waitForReady)
      .then(() => promiseMapSeries(arr, function (val, idx) {
        const step = stepMap(val, idx);

        if (_.isString(step) && !progress.notifyStart(step)) {
          return Promise.reject();
        }

        return Promise.resolve(valueMap(val, idx, progress))
          .then(val => {
            progress.onValue(val, idx);
            return val;
          });
      }))
      .finally(progress.close);
  };
}
