import { EntityType } from 'ui/kibi/components/ontology/entity_type';

import { ProgressMapProvider } from 'ui/kibi/modals/progress_map';

import Promise from 'bluebird';
import _ from 'lodash';


let autoRelationsCache = null;

/**
 * Strips input relation of auto-relations metadata for insertion
 * in the ES backend.
 */
function strippedAutoRelationForOntologySave(relation) {
  return relation.autoRelation
    // Identifiers for autorelations are fake, so we have to remove them.
    // Also, properties 'autoRelation' and 'inverse' are autorelations
    // properties, so we strip them too.
    ? _.omit(relation, 'autoRelation', 'id', 'inverseOf', 'inverse')
    : relation;
}

function strippedAutoEntityForOntologySave(entity) {
  // Removing properties added only for autorelated entities.
  return _.omit(entity, 'autoRelation');
}

function strippedForJson(object) {
  // We're removing internal/angularjs properties added for whatever reason.
  // The rule is that properties starting with $ will not be output in the
  // sessio-stored JSON.
  return _.pick(object, (val, key) => !key.startsWith('$'));
}

// autorelations functions
function _storeEntityId(memo, entityId) {
  memo[entityId] = true;
  return memo;
}

function _formatted(autoRelationsData) {
  autoRelationsData = _.assign({
    entities: [],
    relations: []
  }, autoRelationsData);

  const entityIds = _.map(autoRelationsData.entities, 'id');

  autoRelationsData.isAutoRelatedByEntityId = _(autoRelationsData.relations)
    .map(rel => [rel.domain.id, rel.range.id])
    .flatten()
    .concat(entityIds)
    .reduce(function storeEntityId(memo, entityId) {
      memo[entityId] = true;
      return memo;
    }, {});

  return autoRelationsData;
}

function getAutoRelationsData(sessionStorage) {
  if (!autoRelationsCache) {
    autoRelationsCache = _.assign({
      entities: [],
      relations: [],
      isAutoRelatedByEntityId: {}
    }, sessionStorage.get('autoRelationsData'));
  }

  return autoRelationsCache;
}

function setAutoRelationsData(autoRelationsData, sessionStorage) {
  autoRelationsData = _formatted(autoRelationsData);

  sessionStorage.set('autoRelationsData', autoRelationsData);
  autoRelationsCache = autoRelationsData;
}


function wrapMethods(Model, methodSpecs) {
  // Write operations should always happen against the wrapper, since they just
  // include additional manipulation of the temporary auto-relations should they happen
  // to be the written ones.

  // Read operations depend on the context they are performed - in the management pages
  // we want to show auto-relations, while outside we want them hidden.
  _.assign(Model.prototype, _(methodSpecs.onManagementPage)
    .indexBy()
    .mapValues(methodName => {
      const method = Model.prototype[methodName];

      return function pageChecked(...args) {
        return this.$location.path().startsWith('/management/')
          ? method.call(this, ...args)
          : this.model[methodName](...args);
      };
    }).value());

  // Operations unrelated to temporary relations will pass-through directly
  _.assign(Model.prototype, _(methodSpecs.passThrough)
    .indexBy()
    .mapValues(methodName => function passThrough(...args) {
      return this.model[methodName](...args);
    }).value());
}


class OntologyModelWrapper {
  constructor(Private, services) {
    this.progressMap = Private(ProgressMapProvider);
    this.model = services.ontologyModel;

    _.assign(this, services);
  }

  getRelationList() {
    // Add auto-relations at the end
    const autoRelations = getAutoRelationsData(this.sessionStorage).relations;

    return this.ontologyModel.getRelationList()
      .then(relations => {
        const relationList = relations.concat(autoRelations);

        Object.freeze(relationList);
        return relationList;
      });
  }

  getRelationMap() {
    return this.ontologyModel.getRelationMap()
      .then(relationMap => {
        const newRelationMap = _.clone(relationMap);

        // Append autorelations
        const autoRelations = getAutoRelationsData(this.sessionStorage).relations;

        _.each(autoRelations, autoRel => {
          newRelationMap[autoRel.id] = autoRel;
        });

        Object.freeze(newRelationMap);
        return newRelationMap;
      });
  }

  getRangesForEntityId(entityId) {
    // Concat with autorel search
    const autoRelations = getAutoRelationsData(this.sessionStorage).relations;

    const autoRelRanges = _(autoRelations)
      .filter(relation => relation.domain.id === entityId)
      .map(rel => _.pick(rel.range, 'id', 'type'))
      .value();

    return this.ontologyModel.getRangesForEntityId(entityId)
      .then(ranges => {
        const newRanges = ranges.concat(autoRelRanges);

        Object.freeze(newRanges);
        return newRanges;
      });
  }

  getRelationsByDomain(domainId) {
    // Concat with autorel search
    const autoRelations = getAutoRelationsData(this.sessionStorage).relations;

    const autoRelByDomain = _(autoRelations)
      .filter(relation => relation.domain.id === domainId)
      .map(_.cloneDeep)
      .value();

    return this.ontologyModel.getRelationsByDomain(domainId)
      .then(relations => {
        const newRelations = relations.concat(autoRelByDomain);

        Object.freeze(newRelations);
        return newRelations;
      });
  }

  getUniqueRelationLabels() {
    // Merge list
    const autoRelations = getAutoRelationsData(this.sessionStorage).relations;

    let uniqueAutoRelLabels = _.reduce(autoRelations, (memo, rel) => {
      memo[rel.directLabel] = true;
      memo[rel.inverseLabel] = true;

      return memo;
    }, {});

    uniqueAutoRelLabels = _.keys(uniqueAutoRelLabels);

    return this.ontologyModel.getUniqueRelationLabels()
      .then(uniqueLabels => {
        const uniqueRelationLabels = _.union(uniqueLabels, uniqueAutoRelLabels);

        Object.freeze(uniqueRelationLabels);
        return uniqueRelationLabels;
      });
  }

  getUniqueRelationLabelPairs() {
    // Merge list
    const autoRelations = getAutoRelationsData(this.sessionStorage).relations;

    const uniqueAutoRelLabelPairs = _.reduce(autoRelations, (total, rel) => {
      total.push({
        directLabel : rel.directLabel,
        inverseLabel: rel.inverseLabel
      });
      return total;
    }, []);

    return this.ontologyModel.getUniqueRelationLabelPairs()
      .then(uniqueLabelPairs => {
        const relationLabelPairs = uniqueLabelPairs.concat(uniqueAutoRelLabelPairs);

        const uniqueRelationLabelPairs = _.uniq(relationLabelPairs, pair => {
          return pair.directLabel + pair.inverseLabel;
        });

        Object.freeze(uniqueRelationLabelPairs);
        return uniqueRelationLabelPairs;
      });
  }

  getEntityList() {
    const autoRelationsData = getAutoRelationsData(this.sessionStorage);
    const autoEntities = autoRelationsData.entities;

    return this.ontologyModel.getEntityList()
      .then(entities => {
        const newEntityList = _(entities)
          .concat(autoEntities)
          .map(entity => {
            // Add relevant properties. Input entities from ontologyModel
            // can "mostly" be modified, since they are deep-cloned from cache anyway

            // This added property must not enumerable, so it will not be serialized.
            // It needs to be configurable, however, due to ontologyModel's use
            // of _.throttle - meaning that the same input entity can be received
            // multiple times, the property must be resettable
            Object.defineProperty(entity, 'hasAutoRelations', {
              configurable: true,
              get() {
                return (
                  this.autoRelation ||
                  autoRelationsData.isAutoRelatedByEntityId[this.id]);
              }
            });

            return entity;
          })
          .value();

        Object.freeze(newEntityList);
        return newEntityList;
      });
  }

  getEntityMap() {
    // Prepend autorel search
    const autoEntities = getAutoRelationsData(this.sessionStorage).entities;

    return this.ontologyModel.getEntityMap()
      .then(entityMap => {
        const newEntityMap = _.clone(entityMap);
        _.each(autoEntities, autoEntity => {
          newEntityMap[autoEntity.id] = autoEntity;
        });

        Object.freeze(newEntityMap);
        return newEntityMap;
      });
  }

  getBrokenSavedSearches() {
    return this.ontologyModel.getBrokenSavedSearches();
  }
}

wrapMethods(OntologyModelWrapper, {
  onManagementPage: [
    'getEntityList',
    'getEntityMap',
    'getRelationList',
    'getRelationMap',
    'getRelationsByDomain',
    'getUniqueRelationLabels',
    'getUniqueRelationLabelPairs',
    'getRangesForEntityId',
    'getBrokenSavedSearches'
  ],
  passThrough: [
    'getAllPaths',
    'clearCache'
  ]
});


class DataModelWrapper {
  constructor(Private, services) {
    this.progressMap = Private(ProgressMapProvider);
    this.model = services.dataModel;

    _.assign(this, services);
  }


  // AutoRelations functions

  /**
   * Stores the specified temporary relational configuration in the
   * sessionStorage.
   *
   * @param {Object} autoRelationsData
   *    Auto-generated relational configuration
   */
  setAutoRelationsData(autoRelationsData) {
    return setAutoRelationsData(autoRelationsData, this.sessionStorage);
  }

  /**
   * Returns the temporary relational configuration
   */
  getAutoRelationsData() {
    return getAutoRelationsData(this.sessionStorage);
  }

  /**
   * Clears the temporary relational configuration
   */
  clearAutoRelations() {
    this.sessionStorage.remove('autoRelationsData');
    autoRelationsCache = null;
  }

  /**
   * Updates a session-stored virtual temporary entity.
   *
   * @param {Entity}      entity    Temporary entity to save in the session storage
   */
  updateAutoRelatedEntity(entity) {
    const autoRelationsData = this.getAutoRelationsData();
    const { entities } = autoRelationsData;

    const foundEntityIndex = _.findIndex(entities, ent => ent.id === entity.id);
    if (foundEntityIndex < 0) {
      throw new Error(`Auto-generated temporary entity ${entity.id} not found`);
    }

    entities[foundEntityIndex] = strippedForJson(entity);

    this.setAutoRelationsData(autoRelationsData);
  }

  /**
   * Updates the session-stored auto-relations of the specified entity using
   * data from the input list of relations.
   *
   * This is typically used when a user changes the displayed copy of the
   * temporary relations.
   *
   * @param {Entity}      entity     Entity whose temporary relations must be updated
   * @param {Relation[]}  relations  Temporary relations to save to temporary storage
   */
  updateEntityAutoRelations(entity, relations) {
    const autoRelationsData = getAutoRelationsData(this.sessionStorage);

    const autoRelationsByEntity = _.partition(autoRelationsData.relations,
      rel => rel.range.id === entity.id || rel.domain.id === entity.id);

    const entAutoRelsById = _.indexBy(autoRelationsByEntity[0], 'id');
    const newEntAutoRelsById = {};

    // Apply updated data
    _.forEach(relations, rel => {
      if (!entAutoRelsById[rel.id]) { return; }

      const inverse = entAutoRelsById[rel.inverseOf];
      inverse.domain = rel.range;
      inverse.range = rel.domain;
      inverse.directLabel = rel.inverseLabel;
      inverse.inverseLabel = rel.directLabel;

      newEntAutoRelsById[rel.id] = strippedForJson(rel);
      newEntAutoRelsById[inverse.id] = strippedForJson(inverse);
    });

    autoRelationsData.relations = autoRelationsByEntity[1].concat(
      _.values(newEntAutoRelsById));

    setAutoRelationsData(autoRelationsData, this.sessionStorage);
  }


  // DataModel wrapped functions

  deleteRelationsByDomainOrRange(entityId, ...args) {
    const autoRelationsData = getAutoRelationsData(this.sessionStorage);
    const filteredAutoRelations = autoRelationsData.relations
      .filter(rel => rel.domain.id !== entityId && rel.range.id !== entityId);

    setAutoRelationsData({
      entities: autoRelationsData.entities,
      relations: filteredAutoRelations
    }, this.sessionStorage);

    return this.dataModel.deleteRelationsByDomainOrRange(entityId, ...args);
  }

  deleteEntityIdentifier(entityToDelete, ...args) {
    const { id: entityId } = entityToDelete;

    const autoRelationsData = getAutoRelationsData(this.sessionStorage);
    const autoEntities = autoRelationsData.entities;

    const autoEntity = _.find(autoEntities, ent => ent.id === entityId);

    const deletePromise = autoEntity
      ? Promise.resolve()
      : this.dataModel.deleteEntityIdentifier(entityToDelete, ...args);

    return deletePromise
      .then(() => {
        const filteredEntities = _.without(autoEntities, autoEntity);
        const filteredAutoRelations = autoRelationsData.relations.filter(
          rel => rel.domain.id !== entityId && rel.range.id !== entityId);

        setAutoRelationsData({
          entities: filteredEntities,
          relations: filteredAutoRelations
        }, this.sessionStorage);
      });
  }

  deleteSearch(entity, ...args) {
    const { id: entityId } = entity;

    const autoRelationsData = getAutoRelationsData(this.sessionStorage);

    return this.dataModel.deleteSearch(entity, ...args)
      .then(() => {
        const filteredAutoRelations = autoRelationsData.relations.filter(
          rel => rel.domain.id !== entityId && rel.range.id !== entityId
        );

        setAutoRelationsData({
          entities: autoRelationsData.entities,
          relations: filteredAutoRelations
        }, this.sessionStorage);
      });
  }

  updateEntityIdentifier(entity, ...args) {
    if (!entity.autoRelation) {
      return this.dataModel.updateEntityIdentifier(entity, ...args);
    }

    const autoRelations = getAutoRelationsData(this.sessionStorage).relations;
    const entityAutoRelations =
      _.filter(autoRelations, rel => rel.domain.id === entity.id);

    const opts = { entities: [ entity ] };

    // TODO: It's improper to save associated relations here. However, saving
    // the relations from outside passing through
    // this.dataModel.triggerBeforeSave() is not an option, because the
    // relations that would be saved from there haven't been updated with the
    // new entity id yet.

    return this.createRelations(entityAutoRelations, opts)
      .then(() => {
        const { savedAutoEntityIds } = opts;
        return savedAutoEntityIds[entity.id];
      });
  }

  _applyEid(ent, args) {
    return this.dataModel.createEntityIdentifier(
      strippedAutoEntityForOntologySave(ent)
    ).then(savedId => {
      const { autoRelationsData, savedAutoEntityIds } = args;

      const relations = args.relations.concat(autoRelationsData.relations);

      _.forEach(relations, rel => {
        if (rel.domain.id === ent.id) {
          rel.domain.id = savedId;
        }
        if (rel.range.id === ent.id) {
          rel.range.id = savedId;
        }
      });

      savedAutoEntityIds[ent.id] = savedId;

      delete ent.autoRelation;

      return savedId;
    });
  }

  _applyRelation(relation, args) {
    if (!relation.autoRelation) {
      return this.dataModel.createRelation(relation);
    }

    const { autoRelationsData, savedAutoEntityIds, savedAutoRelationIds } = args;

    const entitiesToSave = _.filter(autoRelationsData.entities, ent =>
      ent.autoRelation &&
      (ent.id === relation.domain.id || ent.id === relation.range.id));

    return Promise.map(entitiesToSave, ent => this._applyEid(ent, args))
      .then(() => this.dataModel.createRelation(
        strippedAutoRelationForOntologySave(relation)))
      .then(relIdPair => {
        savedAutoRelationIds[relation.id] = relIdPair[0];
        savedAutoRelationIds[relation.inverseOf] = relIdPair[1];

        delete relation.autoRelation;

        return relIdPair;
      });
  }

  // Exclude saved entities from autoRel data
  _filterSavedObjects(autoRelationsData, args) {
    setAutoRelationsData({
      entities: autoRelationsData.entities
        .filter(ent => !args.savedAutoEntityIds[ent.id]),
      relations: autoRelationsData.relations
        .filter(rel => !args.savedAutoRelationIds[rel.id])
    }, this.sessionStorage);
  }

  /**
   * Saves the session-backed list of auto-relations into actual ES-backed relations
   *
   * @returns Object
   *    An object containing the list of saved relations and maps of temporary
   *    object ids to saved object ids.
   */
  applyAutoRelations(relations, opts = {}) {
    const { entities = [], showProgress = true } = opts;

    const autoRelationsData = getAutoRelationsData(this.sessionStorage);

    if (!relations) {
      relations = _.filter(autoRelationsData.relations, rel => !rel.inverse);
    } else {
      relations = _.map(relations, rel => {
        if (!rel.autoRelation) { return rel; }

        // Clone relation due to eid saving modifying the ids being referenced
        rel = _.clone(rel);
        rel.domain = _.clone(rel.domain);
        rel.range = _.clone(rel.range);

        return rel;
      });
    }

    const args = {
      autoRelationsData,
      relations,
      savedAutoEntityIds: {},
      savedAutoRelationIds: {}
    };

    return Promise.map(entities, ent => this._applyEid(ent, args))
      .then(() => this.progressMap(relations, {
        showProgress,
        valueMap: rel => this._applyRelation(rel, args),
        stepMap: (rel, r) => `Saving relations (${r} / ${relations.length})`
      }))
      .then(savedRelationIdPairs => ({
        savedRelationIdPairs,
        savedAutoEntityIds: args.savedAutoEntityIds,
        savedAutoRelationIds: args.savedAutoRelationIds
      }))
      .finally(() => {
        this._filterSavedObjects(autoRelationsData, args);

        opts.savedAutoEntityIds = args.savedAutoEntityIds;
        opts.savedAutoRelationIds = args.savedAutoRelationIds;
      });
  }

  // DO NOT IMPLEMENT - Enforces use of plural version instead
  //createRelation(rel) {}

  createRelations(relations, opts = {}) {
    return this.applyAutoRelations(relations, _.assign(opts, { showProgress: false }))
      .then(savedData => savedData.savedRelationIdPairs);
  }

  // DO NOT IMPLEMENT - Enforces use of plural version instead
  //updateRelation(rel) {}

  updateRelations(relations, opts = {}) {
    const relationsByIsNormal = _.partition(relations, 'autoRelation');

    return Promise.all([
      this.applyAutoRelations(relationsByIsNormal[0],
        _.assign(opts, { showProgress: false })),
      this.dataModel.updateRelations(relationsByIsNormal[1])
    ]);
  }

  // DO NOT IMPLEMENT - Enforces use of plural version instead
  //deleteRelation(relId) {}

  deleteRelations(relationIds) {
    const autoRelationsData = getAutoRelationsData(this.sessionStorage);
    const relationIdsHash = _.indexBy(relationIds);

    const autoRelationIdsHash = _.reduce(autoRelationsData.relations, (memo, rel) => {
      if (relationIdsHash[rel.id]) {
        memo[rel.id] = true;
        delete relationIdsHash[rel.id];

        memo[rel.inverseOf] = true;
        delete relationIdsHash[rel.inverseOf];
      }

      return memo;
    }, {});

    const filteredAutoRelations = _.filter(autoRelationsData.relations,
      rel => !autoRelationIdsHash[rel.id]);

    const actualRelationsToRemove = _.values(relationIdsHash);

    setAutoRelationsData({
      entities: autoRelationsData.entities,
      relations: filteredAutoRelations
    }, this.sessionStorage);

    return this.dataModel.deleteRelations(actualRelationsToRemove);
  }
}

wrapMethods(DataModelWrapper, {
  onManagementPage: [],
  passThrough: [
    'registerBeforeSave',
    'deregisterBeforeSave',
    'triggerBeforeSave',
    'getRootSearchId',
    'getSearchEntityToInsert',
    'getEntityIdentifierToInsert',
    'getEntityForUpdate',
    'getIndexPropertiesToInsert',
    'getIndexPatternSearchEntityToInsert',
    'findIndexPatternSearchByPattern',
    'createIndexPatternSearch',
    'createSearch',
    'createEntityIdentifier',
    'updateSearch'
  ]
});

/**
 * OntologyWrapper wraps an OntologyModel/DataModel instance to inject
 * temporary auto-generated relations and entities.
 *
 * The temporary relations/entities are available *only* on '/management/'
 * pages and are stored in the browser's sessionStorage until they are
 * explicitly inserted in the ontology, at which point they are removed from
 * sessionStorage and inserted as actual relations in the wrapped
 * ontologyModel.
 *
 * Typical usage:
 *    ontologyModel = Private(OntologyWrapperProvider).forOntologyModel(ontologyModel);
 *    dataModel = Private(OntologyWrapperProvider).forDataModel(dataModel);
 */
export function OntologyWrapperProvider(Private, sessionStorage, $location) {
  class OntologyWrapper {
    forOntologyModel(ontologyModel) {
      return new OntologyModelWrapper(Private, { ontologyModel, sessionStorage, $location });
    }

    forDataModel(dataModel) {
      return new DataModelWrapper(Private, { dataModel, sessionStorage, $location });
    }
  }

  return new OntologyWrapper();
}
