import _ from 'lodash';
import qs from 'ui/utils/query_string';
import { parseWithPrecision } from 'ui/kibi/utils/date_math_precision';
import { uniqFilters } from 'ui/filter_bar/lib/uniq_filters';
import { toJson } from 'ui/utils/aggressive_parse';
import angular from 'angular';
import { onManagementPage, onDashboardPage, onVisualizePage } from 'ui/kibi/utils/on_page';
import { uiModules } from 'ui/modules';
import uiRoutes from 'ui/routes';
import { StateProvider } from 'ui/state_management/state';
import { getAppUrl, getBasePath } from 'ui/chrome';
import { IndexPatternMissingIndices } from 'ui/errors';
import { DecorateQueryProvider } from 'ui/courier/data_source/_decorate_query';
import moment from 'moment';
import {
  findMainCoatNodeSearchId,
  findMainCoatNode
} from 'ui/kibi/components/dashboards360/coat_tree';
import { filterHelper } from 'ui/kibi/components/dashboards360/filter_helper';
import { coatHelper } from 'ui/kibi/components/dashboards360/coat_helper';

// kibi: this import is needed for any third party plugin
import 'ui/saved_objects_api';
import 'ui/courier/courier';
import 'ui/index_patterns/index_patterns';
import 'ui/kibi/mappings';
import 'plugins/kibana/visualize/saved_visualizations/saved_visualizations'; // this one is needed because it is used by coatDashboardMap
import 'plugins/kibana/dashboard/saved_dashboard/saved_dashboards';
import 'plugins/kibana/discover/saved_searches/saved_searches';
import 'plugins/kibana/discover/saved_eids/saved_eids';
import 'plugins/kibana/discover/saved_relations/saved_relations';
import 'ui/kibi/components/ontology_model/ontology_model';
import 'ui/kibi/components/query_engine_client/query_engine_client';
import 'ui/kibi/components/dashboards360/coat_dashboard_map';
import 'ui/kibi/components/dashboards360/use_visualization_for_count_on_dashboard';
import { isDefaultQuery, getDefaultQuery } from 'ui/parse_query';
// kibi: end

function KibiStateProvider(savedSearches, timefilter, $route, Promise, getAppState, savedDashboards, $rootScope, indexPatterns, globalState,
  elasticsearchPlugins, $location, config, Private, createNotifier, coatDashboardMap) {
  const State = Private(StateProvider);
  const notify = createNotifier({ location: 'Siren State' });
  const decorateQuery = Private(DecorateQueryProvider);

  _.class(KibiState).inherits(State);
  function KibiState(defaults) {
    KibiState.Super.call(this, '_k', defaults);

    this.init = _.once(function () {
      // do not try to initialize the kibistate if it was already done via the URL
      if (_.size(this.toObject())) {
        return;
      }
      return savedDashboards.find().then((resp) => {
        if (resp.hits) {
          _.each(resp.hits, (dashboard) => {
            const meta = JSON.parse(dashboard.kibanaSavedObjectMeta.searchSourceJSON);
            let filters = _.reject(meta.filter, this._isNotDashFilter);
            const query = _.find(meta.filter, (filter) => filter.query && filter.query.query_string && !filter.meta);

            // query
            if (query && query.query && !this._isDefaultQuery(query.query)) {
              this._setDashboardProperty(dashboard.id, this._properties.query, query.query);
            }

            filters = JSON.parse(toJson(filters, angular.toJson));
            if (filters && filters.length) {
              this._setDashboardProperty(dashboard.id, this._properties.filters, filters);
            }
            // time
            if (dashboard.timeRestore && dashboard.timeFrom && dashboard.timeTo) {
              this._saveTimeForDashboardId(dashboard.id, dashboard.timeMode, dashboard.timeFrom, dashboard.timeTo);
            }
          });

          this.save(true, true);
        }
      }).catch(notify.error);
    });
  }

  /**
   * isFilterOutdated returns true if the given filter uses an outdated API
   */
  KibiState.prototype.isFilterOutdated = function (filter) {
    return filter && filter.join_sequence && !filter.meta.version;
  };

  /*
  * _isNotDashFilter returns true if the filter is not a dashboard filter
  */
  KibiState.prototype._isNotDashFilter = (filter) => {
    return filter.query && (filter.query.match_all || filter.query.query_string) && !filter.meta;
  };

  /**
   * disableFiltersIfOutdated disables filters which rely on outdated API and warns the user about it.
   *
   * @param filters a list of filters
   * @param dashboardId the dashboard ID on which the filters are
   */
  KibiState.prototype.disableFiltersIfOutdated = function (filters, dashboardId) {
    const appState = getAppState();

    if (!dashboardId) {
      throw new Error('disableFiltersIfOutdated called without a dashboardId');
    }
    _.each(filters, filter => {
      if (this.isFilterOutdated(filter)) {
        let message;

        if (dashboardId === this.getCurrentDashboardId() && appState && !_.find(appState.filters, filter)) {
          // the filter is not in the appState, the KibiState is then dirty
          message = `The Kibi state contains filters that rely on outdated API. Please clean it, either by going to ${getAppUrl()},
            or by switching to another dashboard.`;
        } else {
          message = `The join filter "${filter.meta.alias}" on dashboard with ID="${dashboardId}" is invalid ` +
            'because it relies on outdated API. Please remove it.';
        }
        message += ` If the filter keeps on coming back, then it may be saved with the dashboard with ID="${dashboardId}"`;
        notify.warning(message);

        filter.meta.disabled = true;
      }
    });
  };

  // if the url param is missing, write it back
  KibiState.prototype._persistAcrossApps = true;

  KibiState.prototype.removeFromUrl = function (url) {
    return qs.replaceParamInUrl(url, this._urlParam, null);
  };

  KibiState.prototype._isDefaultQuery = isDefaultQuery;

  /**
   * Returns true if the given time is the one from timepicker:timeDefaults
   */
  KibiState.prototype._isDefaultTime = function (mode, from, to) {
    const timeDefaults = config.get('timepicker:timeDefaults');
    return mode === timeDefaults.mode && from === timeDefaults.from && to === timeDefaults.to;
  };

  /**
   * Saves the given time to the kibistate
   */
  KibiState.prototype._saveTimeForDashboardId = function (dashboardId, mode, from, to) {
    let toStr = to;
    let fromStr = from;

    if (typeof from === 'object') {
      fromStr = from.toISOString();
    }
    if (typeof to === 'object') {
      toStr = to.toISOString();
    }
    const oldTime = this._getDashboardProperty(dashboardId, this._properties.time);
    const changed = this._setDashboardProperty(dashboardId, this._properties.time, {
      m: mode,
      f: fromStr,
      t: toStr
    });
    if (changed && this.getCurrentDashboardId() !== dashboardId) {
      // do not emit the event if the time changed is for the current dashboard since this is taken care of by the globalState
      const newTime = this._getDashboardProperty(dashboardId, this._properties.time);
      this.emit('time', dashboardId, newTime, oldTime);
    }
    return changed;
  };

  /**
   * Shortcuts for properties in the kibistate
   */
  KibiState.prototype._properties = {
    // dashboards properties
    filters: 'f',
    query: 'q',
    time: 't',
    synced_dashboards: 's',
    // properties available in the diff array with the save_with_changes event
    dashboards: 'd',
    groups: 'g',
    // selected entity properties
    selected_entity_disabled: 'x',
    selected_entity: 'u',
    test_selected_entity: 'v'
  };

  /**
   * setSyncedDashboards sets the given dashboard IDs to sync the time with the given dashboard.
   */
  KibiState.prototype.setSyncedDashboards = function (dashboardId, dashboards) {
    if (dashboards && dashboards.length) {
      this._setDashboardProperty(dashboardId, this._properties.synced_dashboards, dashboards);
    } else {
      this._deleteDashboardProperty(dashboardId, this._properties.synced_dashboards);
    }
  };

  /**
   * getSyncedDashboards returns the IDs of dashboards to sync the time with.
   */
  KibiState.prototype.getSyncedDashboards = function (dashboardId) {
    return this._getDashboardProperty(dashboardId, this._properties.synced_dashboards);
  };

  KibiState.prototype.isEntitySelected = function (index, type, id, column) {
    const entityURI = this.getEntityURI();
    if (!entityURI || !index || !type || !id || !column) {
      return false;
    }
    return entityURI.index === index && entityURI.type === type && entityURI.id === id && entityURI.column === column;
  };

  KibiState.prototype.setEntityURI = function ({ index, type, id, column } = {}) {
    if (onDashboardPage()) {
      if (!id) {
        delete this[this._properties.selected_entity];
      } else {
        this[this._properties.selected_entity] = { index, type, id, column };
      }
    } else if (onVisualizePage() || onManagementPage()) {
      if (!id) {
        delete this[this._properties.test_selected_entity];
      } else {
        this[this._properties.test_selected_entity] = { index, type, id, column };
      }
    } else {
      throw new Error('Cannot set entity URI because you are not on dashboard/visualize/management');
    }
  };

  KibiState.prototype.getEntityURI = function () {
    if (onDashboardPage()) {
      return this[this._properties.selected_entity];
    } else if (onVisualizePage() || onManagementPage()) {
      return this[this._properties.test_selected_entity];
    }
    throw new Error('Cannot get entity URI because you are not on dashboard/visualize/management');
  };

  KibiState.prototype.isSelectedEntityDisabled = function () {
    return Boolean(this[this._properties.selected_entity_disabled]);
  };

  KibiState.prototype.disableSelectedEntity = function (disable) {
    if (disable) {
      this[this._properties.selected_entity_disabled] = disable;
    } else {
      delete this[this._properties.selected_entity_disabled];
    }
  };

  KibiState.prototype.removeTestEntityURI = function () {
    delete this[this._properties.test_selected_entity];
  };

  KibiState.prototype._compareFilters = function (filter1, filter2) {
    const f1 = _.cloneDeep(filter1);
    const f2 = _.cloneDeep(filter2);
    delete f1.meta;
    delete f2.meta;
    delete f1.$state;
    delete f2.$state;
    delete f1.$$hashKey;
    delete f2.$$hashKey;
    return angular.equals(f1,f2);
  };

  // checks if all objects from "b" are present in "a"
  KibiState.prototype._arrayContainsArray = function (a, b) {
    return _.every(b, bObject => {
      const found =  _.find(a, aObject => {
        return this._compareFilters(aObject, bObject);
      });
      return found;
    });
  };

  KibiState.prototype._elementsFromFirstNotInSecond = function (a, b) {
    return _.filter(a, aObject => {
      const found =  _.find(b, bObject => {
        return this._compareFilters(aObject, bObject);
      });
      return !found;
    });
  };

  KibiState.prototype._getGlobalTime = function (index) {
    if (!globalState.time) {
      return;
    }
    const time = {
      mode: globalState.time.mode,
      from: globalState.time.from,
      to: globalState.time.to
    };
    return time;
  };

  KibiState.prototype._translateKibiStateTime = function (time) {
    return {
      from: time.f,
      to: time.t,
      mode: time.m
    };
  };

  KibiState.prototype._filterOutDisabled = function (filters) {
    return _.filter(filters, filter => {
      return !filter.meta || (filter.meta && !filter.meta.disabled);
    });
  };


  // Compare stateA to stateB where stateB is the main source of truth
  // returns a diff object with missing and extra properties
  // missing - constains all that is missing from stateB
  // extra - constains all that is extra (not in stateB)
  //
  // stateA and stateB are in a form of
  // {
  //   filters: [],
  //   query: {}
  // }
  KibiState.prototype.compareStates = function (stateA, stateB) {
    const resp = {
      stateEqual: true,
      diff: {
        extra: {
          filters: [],
          query: null
        },
        missing: {
          filters: [],
          query: null
        }
      }
    };

    const aFilters = this._filterOutDisabled(stateA.filters || []);
    const bFilters = this._filterOutDisabled(stateB.filters || []);
    const aQuery = stateA.query || getDefaultQuery();
    const bQuery = stateB.query || getDefaultQuery();

    const aContainsBFilters = this._arrayContainsArray(aFilters, bFilters);
    const aQueryEqualBQuery = angular.equals(aQuery, bQuery);
    const aStateContainsBState = aContainsBFilters && aQueryEqualBQuery;
    const aStateContainsAnExtraFilter = aStateContainsBState && aFilters.length > bFilters.length;

    resp.diff.extra.filters = this._elementsFromFirstNotInSecond(aFilters, bFilters);
    resp.diff.missing.filters = this._elementsFromFirstNotInSecond(bFilters, aFilters);

    const isAQueryDefault = angular.equals(aQuery, getDefaultQuery());
    if (!aQueryEqualBQuery && isAQueryDefault) {
      resp.diff.missing.query = aQuery;
    }
    if (!aQueryEqualBQuery && !isAQueryDefault) {
      resp.diff.extra.query = aQuery;
    }

    resp.stateEqual = aStateContainsBState && !aStateContainsAnExtraFilter;

    return resp;
  };

  KibiState.prototype._compareTimes = function (timeA, timeB) {
    if (!timeA || !timeB) {
      // Input times can be undefined (e.g. global state time from a new ES installation)
      return timeA === timeB;
    }
    if (timeA.mode !== timeB.mode) {
      return false;
    }
    if (timeA.mode === timeB.mode && timeB.mode === 'absolute') {
      // first convert to moments
      const fromA = moment(timeA.from).valueOf();
      const toA = moment(timeA.to).valueOf();
      const fromB = moment(timeB.from).valueOf();
      const toB = moment(timeB.to).valueOf();
      return fromA === fromB && toA === toB;
    }

    return angular.equals(timeA, timeB);
  };

  // I think dashboard should be marked in the following way
  // dirty equals to:
  //
  // For current dashbaord
  //
  // false - if appState contains all dashState
  //         AND
  //         if there is dashTime, dashTime is equal globalTime
  //
  // true  - if appState does NOT contain all dashState
  //         OR
  //         appState contains some extra state not present in dashState
  //         OR
  //         if there is dashTime, dashTime is NOT equal globalTime
  //
  // For dashbaord which is NOT current dashboard
  //
  // false - if kibiState contains all dashState
  //         AND
  //         if there is dashTime, dashTime is equal to time stored in kibiState
  //
  // true  - if akibiState does NOT contain all dashState
  //         OR
  //         kibiState contains some extra state not present in dashState
  //         OR
  //         if there is dashTime, dashTime is NOT equal time stored in kibiState
  KibiState.prototype.compareStateToSavedDashState = async function (
    dashboard
  ) {
    const resp = {
      stateEqual: true,
      diff: {
        extra: {
          filters: [],
          query: null
        },
        missing: {
          filters: [],
          query: null
        },
        timeIsDifferent: false
      }
    };

    const appState = getAppState();
    const currentDashId = this.getCurrentDashboardId();

    const dashFilters = this._filterOutDisabled(
      this.getSavedDashFilters(dashboard)
    );
    const dashQuery = this.getSavedDashQuery(dashboard) || getDefaultQuery();

    if (dashboard.id === currentDashId) {
      const appStateQuery = (appState && appState.query) ? appState.query : getDefaultQuery();
      let appStateFilters;
      if (appState && appState.filters) {
        appStateFilters = appState.filters;
      } else if (this.d && this.d[dashboard.id] && this.d[dashboard.id].f) {
        // here we also check if there are some filters in the kibi state for this dashboard
        // it might happen that they were not yet copied into appState or appState was not yet present
        // in such case this function was giving wrong result
        appStateFilters = this.d[dashboard.id].f;
      } else {
        appStateFilters = [];
      };
      appStateFilters = this._filterOutDisabled(appStateFilters);

      const appStateFiltersContainDashFilters = this._arrayContainsArray(appStateFilters, dashFilters);
      const appStateQueryEqualDashQuery = angular.equals(appStateQuery, dashQuery);
      const appStateContainsDashState = appStateFiltersContainDashFilters && appStateQueryEqualDashQuery;
      const appStateContainsAnExtraFilter = appStateFiltersContainDashFilters && appStateFilters.length > dashFilters.length;

      resp.diff.extra.filters = this._elementsFromFirstNotInSecond(appStateFilters, dashFilters);
      resp.diff.missing.filters = this._elementsFromFirstNotInSecond(dashFilters, appStateFilters);

      if (!appStateQueryEqualDashQuery) {
        if (!angular.equals(appStateQuery, getDefaultQuery())) {
          resp.diff.extra.query = appStateQuery;
        }
        if (!angular.equals(dashQuery, getDefaultQuery())) {
          resp.diff.missing.query = dashQuery;
        }
      }

      if (dashboard.timeRestore) {
        const globalTime = this._getGlobalTime();
        const dashTime = this.getSavedDashTime(dashboard);

        const globalTimeEqualDashTime = this._compareTimes(globalTime, dashTime);
        resp.stateEqual = appStateContainsDashState && !appStateContainsAnExtraFilter && globalTimeEqualDashTime;
        if (!globalTimeEqualDashTime) {
          resp.diff.timeIsDifferent = true;
        }
      } else {
        resp.stateEqual = appStateContainsDashState && !appStateContainsAnExtraFilter;
      }
    } else {
      // the kibi state for this dashboard exist
      if (this[this._properties.dashboards] && this[this._properties.dashboards][dashboard.id]) {
        const kibiStateQuery = this._getDashboardProperty(dashboard.id, this._properties.query) || getDefaultQuery();
        const kibiStateFilters = this._filterOutDisabled(
          this._getDashboardProperty(dashboard.id, this._properties.filters) || []
        );

        const kibiStateFiltersContainDashFilters = this._arrayContainsArray(kibiStateFilters, dashFilters);
        const kibiStateQueryEqualDashQuery = angular.equals(kibiStateQuery, dashQuery);
        const kibiStateContainsDashState = kibiStateFiltersContainDashFilters && kibiStateQueryEqualDashQuery;
        const kibiStateContainsAnExtraFilter = kibiStateFiltersContainDashFilters && kibiStateFilters.length > dashFilters.length;

        resp.diff.extra.filters = this._elementsFromFirstNotInSecond(kibiStateFilters, dashFilters);
        resp.diff.missing.filters = this._elementsFromFirstNotInSecond(dashFilters, kibiStateFilters);
        if (!kibiStateQueryEqualDashQuery) {
          resp.diff.extra.query = kibiStateQuery;
        }

        if (dashboard.timeRestore) {
          let kibiStateTime = this._getDashboardProperty(dashboard.id, this._properties.time);

          // could be that user did not yet visited the dashboard so the time is not yet stored in kibiState
          if (kibiStateTime) {
            kibiStateTime = this._translateKibiStateTime(kibiStateTime);
            const dashTime = this.getSavedDashTime(dashboard);
            const kibiStateTimeEqualDashTime = this._compareTimes(kibiStateTime, dashTime);
            resp.stateEqual = kibiStateContainsDashState && !kibiStateContainsAnExtraFilter && kibiStateTimeEqualDashTime;
            if (!kibiStateTimeEqualDashTime) {
              resp.diff.timeIsDifferent = true;
            }
          }
        } else {
          resp.stateEqual = kibiStateContainsDashState && !kibiStateContainsAnExtraFilter;
        }
      }
    }
    return resp;
  };

  /**
   * Reset the filters, queries, and time for each dashboard to their saved state.
   * Added dashId to allow reset only one dashboard.
   */
  KibiState.prototype.resetFiltersQueriesTimes = function (dashId) {
    if (!dashId) {
      if (globalState.filters && globalState.filters.length) {
        // remove pinned filters
        globalState.filters = [];
        globalState.save();
      }
    }
    return savedDashboards.find().then((resp) => {
      if (resp.hits) {
        const dashboardIdsToUpdate = [];
        const appState = getAppState();
        const timeDefaults = config.get('timepicker:timeDefaults');
        // kibi: Do not mutate object received from cache
        let hits = resp.hits;
        if (dashId) {
          hits = _(resp.hits).filter(d => d.id === dashId).value();
        }

        _.each(hits, (dashboard) => {
          const meta = JSON.parse(dashboard.kibanaSavedObjectMeta.searchSourceJSON);
          const filters = _.reject(meta.filter, this._isNotDashFilter);
          const query = _.find(meta.filter, (filter) => filter.query && filter.query.query_string && !filter.meta);

          this.disableFiltersIfOutdated(filters, dashboard.id);
          // reset appstate
          if (appState && dashboard.id === appState.id) {
            let queryChanged = false;
            // filters
            appState.filters = filters;

            // query
            const origQuery = query && query.query || getDefaultQuery();
            if (!angular.equals(origQuery, appState.query)) {
              queryChanged = true;
            }
            appState.query = origQuery;

            // time
            if (dashboard.timeRestore && dashboard.timeFrom && dashboard.timeTo) {
              timefilter.time.mode = dashboard.timeMode;
              timefilter.time.to = dashboard.timeTo;
              timefilter.time.from = dashboard.timeFrom;
            } else {
              // These can be date math strings or moments.
              timefilter.time = timeDefaults;
            }
            if (queryChanged) {
              // this will save the appstate and update the current searchsource
              // This is only needed for changes on query, since the query needs to be added to the searchsource
              this.emit('reset_app_state_query', appState.query);
            } else {
              appState.save();
            }
          }

          // reset kibistate
          let modified = false;
          if (this[this._properties.dashboards] && this[this._properties.dashboards][dashboard.id]) {
            // query
            if (!query || this._isDefaultQuery(query.query)) {
              if (this._getDashboardProperty(dashboard.id, this._properties.query)) {
                // kibistate has a query that will be removed with the reset
                modified = true;
              }
              this._deleteDashboardProperty(dashboard.id, this._properties.query);
            } else {
              if (this._setDashboardProperty(dashboard.id, this._properties.query, query.query)) {
                modified = true;
              }
            }

            // filters
            if (filters.length) {
              if (this._setDashboardProperty(dashboard.id, this._properties.filters, filters)) {
                modified = true;
              }
            } else {
              if (this._getDashboardProperty(dashboard.id, this._properties.filters)) {
                // kibistate has filters that will be removed with the reset
                modified = true;
              }
              this._deleteDashboardProperty(dashboard.id, this._properties.filters);
            }

            // time
            if (dashboard.timeRestore && dashboard.timeFrom && dashboard.timeTo) {
              if (this._saveTimeForDashboardId(dashboard.id, dashboard.timeMode, dashboard.timeFrom, dashboard.timeTo)) {
                modified = true;
              }
            } else {
              if (this._getDashboardProperty(dashboard.id, this._properties.time)) {
                // kibistate has a time that will be removed with the reset
                modified = true;
              }
              this._deleteDashboardProperty(dashboard.id, this._properties.time);
            }
          }
          if (modified) {
            dashboardIdsToUpdate.push(dashboard.id);
          }
        });
        if (dashboardIdsToUpdate.length) {
          this.emit('reset', dashboardIdsToUpdate);
        }
        this.save();
      }
    });
  };

  KibiState.prototype.getSelectedDashboardId = function (groupId) {
    if (this[this._properties.groups]) {
      return this[this._properties.groups][groupId];
    }
    return null;
  };

  KibiState.prototype.setSelectedDashboardId = function (groupId, dashboardId) {
    if (!this[this._properties.groups]) {
      this[this._properties.groups] = {};
    }
    this[this._properties.groups][groupId] = dashboardId;
  };

  KibiState.prototype.addFilter = function (dashboardId, filter) {
    const comparatorOptions = {
      negate: true,
      disabled: true
    };

    const filters = this._getDashboardProperty(dashboardId, this._properties.filters) || [];
    filters.push(filter);
    this._setDashboardProperty(dashboardId, this._properties.filters, uniqFilters(filters, comparatorOptions));
  };

  /**
   * Returns the current dashboard if exists and not locked.
   */
  KibiState.prototype.getDashboardOnView = function () {
    const dash = _.get($route, 'current.locals.dash');

    if (!dash || dash.locked) {
      return;
    }
    return dash;
  };

  /**
   * Returns the current dashboard id.
   * Don't use this function to check if we are on a dashboard. Please use getDashboardOnView instead.
   */
  KibiState.prototype.getCurrentDashboardId = function () {
    const dash = this.getDashboardOnView();

    if (dash) {
      return dash.id;
    } else {
      // try to get the dashboard id from the current params
      const params = _.get($route, 'current.params');
      if (params && params.id) {
        return params.id;
      }
      return;
    }
  };

  /**
   * Sets a property-value pair for the given dashboard
   *
   * @param dashboardId the ID of the dashboard
   * @param prop the property name
   * @param value the value to set
   * @returns boolean true if the property changed
   */
  KibiState.prototype._setDashboardProperty = function (dashboardId, prop, value) {
    if (!this[this._properties.dashboards]) {
      this[this._properties.dashboards] = {};
    }
    if (!this[this._properties.dashboards][dashboardId]) {
      this[this._properties.dashboards][dashboardId] = {};
    }
    const changed = !angular.equals(this[this._properties.dashboards][dashboardId][prop], value);
    this[this._properties.dashboards][dashboardId][prop] = value;
    return changed;
  };

  /**
   * Gets a property-value pair for the given dashboard
   */
  KibiState.prototype._getDashboardProperty = function (dashboardId, prop) {
    if (!this[this._properties.dashboards] || !this[this._properties.dashboards][dashboardId]) {
      return;
    }
    return this[this._properties.dashboards][dashboardId][prop];
  };

  /**
   * Delets the property from the dashboards object in the kibistate
   */
  KibiState.prototype._deleteDashboardProperty = function (dashboardId, prop) {
    if (!this[this._properties.dashboards] || !this[this._properties.dashboards][dashboardId]) {
      return;
    }
    delete this[this._properties.dashboards][dashboardId][prop];
    // check if this was the last and only
    // if yes delete the whole dashboard object
    if (Object.keys(this[this._properties.dashboards][dashboardId]).length === 0) {
      delete this[this._properties.dashboards][dashboardId];
    }
  };

  /**
   * For each dashboard id in the argument, return a promise with the saved dashboard and associated saved search meta.
   * If dashboardIds is undefined, all dashboards are returned.
   *
   * @param dashboardIds array list of dashboard ids
   * @param failOnMissingMeta boolean if true then an unknown saved search will fail, otherwise a notification is printed and it is skipped
   * @param suppressWarnings boolean if true then do not output warning toast notifications
   * @returns Promise array of dashboard and search pairs
   */
  KibiState.prototype._getDashboardAndSavedSearchMetas = function (dashboardIds, failOnMissingMeta = true, suppressWarnings = false) {
    const getAllDashboards = !dashboardIds;

    dashboardIds = _.compact(dashboardIds);

    // use find to minimize number of requests
    return Promise.all([ savedSearches.find(), savedDashboards.find() ])
      .then(([ savedSearchesRes, savedDashboardsRes ]) => {
        const errors = [];
        const savedDashboardsAndsavedMetas = _(savedDashboardsRes.hits)
        // keep the dashboards that are in the array passed as argument
          .filter((savedDash) => getAllDashboards || _.contains(dashboardIds, savedDash.id))
          .tap(savedDashMetas => {
            if (!getAllDashboards && savedDashMetas.length !== dashboardIds.length) {
              errors.push(`Unable to retrieve dashboards: ${_.difference(dashboardIds, _.map(savedDashMetas, 'id'))}.`);
            }
          })
          .map((savedDash) => {
            savedDash.$$savedSearchId = findMainCoatNodeSearchId(savedDash.coatJSON);
            const savedSearch = _.find(savedSearchesRes.hits, (hit) => hit.id === savedDash.$$savedSearchId);
            const savedSearchMeta = savedSearch ? JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON) : null;
            return { savedDash, savedSearchMeta };
          })
          .sortBy(({ savedDash }) => {
            if (dashboardIds && dashboardIds.length > 0) {
            // here we need to sort the results based on dashboardIds order
              return dashboardIds.indexOf(savedDash.id);
            }
          })
          .value();

        const dashboardsMissingSearch = savedDashboardsAndsavedMetas
          .filter(({ savedSearchMeta, savedDash }) => !savedSearchMeta && savedDash.$$savedSearchId)
          .map(({ savedDash }) => savedDash.title);

        if (dashboardsMissingSearch.length === 1) {
          errors.push(
            `The dashboard [${dashboardsMissingSearch[0]}] is associated with an unknown saved search. ` +
          'The search may have been removed or you may not have the rights to access it.'
          );
        } else if (dashboardsMissingSearch.length > 1) {
          errors.push(
            `The dashboards [${dashboardsMissingSearch.join(', ')}] are associated with unknown saved searches. ` +
          'The searches may have been removed or you may not have the rights to access them.'
          );
        }

        if (errors.length) {
          if (failOnMissingMeta) {
            return Promise.reject(new Error(errors[0])); // take the first error
          } else if (!suppressWarnings) {
            errors.forEach(error => notify.error(error));
          }
        }

        return savedDashboardsAndsavedMetas.filter(({ savedDash }) => !dashboardsMissingSearch.includes(savedDash.title));
      });
  };

  /**
   * Copied from 'ui/filter_bar/query_filter'.
   * Rids filter list of null values and replaces state if any nulls are found.
   * Work around for https://github.com/elastic/kibana/issues/5896.
   */
  function validateStateFilters(state) {
    if (!state.filters) {
      return [];
    }
    const compacted = _.compact(state.filters);
    if (state.filters.length !== compacted.length) {
      state.filters = compacted;
      state.replace();
    }
    return state.filters;
  }

  KibiState.prototype.getSavedDashFilters = function (savedDash) {

    const savedDashMeta = (savedDash.kibanaSavedObjectMeta && savedDash.kibanaSavedObjectMeta.searchSourceJSON) ?
      JSON.parse(savedDash.kibanaSavedObjectMeta.searchSourceJSON) : null;

    if (savedDashMeta && savedDashMeta.filter) {
      // here find and remove the query
      const filters = _.filter(savedDashMeta.filter, filter => {
        return !!filter.meta; // has to have meta - only query is saved without meta
      });
      return filters;
    }
    return [];
  };

  KibiState.prototype.getSavedDashQuery = function (savedDash) {
    const savedDashMeta = (savedDash.kibanaSavedObjectMeta && savedDash.kibanaSavedObjectMeta.searchSourceJSON) ?
      JSON.parse(savedDash.kibanaSavedObjectMeta.searchSourceJSON) : null;
    if (savedDashMeta && savedDashMeta.filter) {
      const query = _.find(savedDashMeta.filter, filter => {
        return filter.query && filter.query.query_string && filter.meta === undefined; // only query does not have meta
      });
      if (query) {
        return query.query;
      }
    }
    return null;
  };

  KibiState.prototype.getSavedDashTime = function (savedDash) {
    if (savedDash && savedDash.timeRestore) {
      const time = {
        mode: savedDash.timeMode,
        from: savedDash.timeFrom,
        to: savedDash.timeTo
      };
      return time;
    }
    return null;
  };

  function addToFilterArray(filterArray, filter, disabled) {
    if (disabled) {
      filterArray.push(filter);
    } else {
      if (!filter.meta.disabled) {
        filterArray.push(filter);
      }
    }
  }

  /**
   * Returns the current set of filters for the given dashboard.
   * If pinned is true, then the pinned filters are added to the returned array.
   * If disabled is true, then the disabled filters are added to the returned array.
   * If injectFwFilter = false no fw computation is performed,
   * otherwise we check if the coat is there and compute the
   * implicit join filter and inject it into returned array,
   * at the same time we make sure that original fw filters are stripped from returned array
   */
  KibiState.prototype._getFilters = function (dashboardId, appState, metas, time, { pinned, disabled, injectFwFilter = true }) {
    let filterCondidates;

    if (appState && this.getCurrentDashboardId() === dashboardId) {
      filterCondidates = _.cloneDeep(validateStateFilters(appState));
    } else {
      const kibiStateFilters = this._getDashboardProperty(dashboardId, this._properties.filters);
      filterCondidates = kibiStateFilters && _.cloneDeep(kibiStateFilters) || [];
    }

    let mainFilters = [];
    if (injectFwFilter === false) {
      // this is the case when we call it from saveAppState
      mainFilters = filterCondidates;
    } else {
      // dashboards360 bit
      // First we should filter out the foreign filters needed to build implicit join filter
      // Next if injectFwFilter === true (default) we turn the foreign filters into a
      // join filter and inject it into main filters instead of original foreign filters
      const coat = coatDashboardMap.getCoatFromCoatDashboardMap(dashboardId);

      // TODO: code similar to the one in _abstract.js should be possible to unified and refactor
      const rootNode = (coat && coat.items) ? findMainCoatNode(coat.items) : undefined;
      _.each(filterCondidates, filter => {
        if (coat && coat.items && _.get(filter, 'meta._siren.vis.id') && _.get(filter, 'meta._siren.vis.panelIndex')) {
          const filterNode = coatHelper.findItemByVisIdAndPanelIndex(
            coat.items, filter.meta._siren.vis.id, filter.meta._siren.vis.panelIndex
          );
          if (!filterNode || !rootNode) {
            addToFilterArray(mainFilters, filter, disabled);
          } else if (filterNode && rootNode) {
            if (filterNode.id === rootNode.id) {
              addToFilterArray(mainFilters, filter, disabled);
            }
          }
        } else {
          addToFilterArray(mainFilters, filter, disabled);
        }
      });
      // TODO: end

      if (coat && coat.items) {
        const isTheRequestFromForeignWidgetDashboard = coatHelper.isItForeingWidgetsCoat({ coat });
        if (isTheRequestFromForeignWidgetDashboard) {
          // only in such case try to build the implicit join and inject it
          const useRootNodeAsFocus = true;

          const foreignSearchJoinFilters = filterHelper.composeJoinFilterForForeignVis({
            _sirenFromVis: { coat },
            allFilters: filterCondidates,
            time,
            sirenTimePrecision: $rootScope.sirenTimePrecision,
            useRootNodeAsFocus // focus node should be the target of the join !!!
          });

          // here filter and get only the join sequence filters
          const joinSequenseFilters = _.filter(foreignSearchJoinFilters, f => f.join_sequence);
          if (joinSequenseFilters.length) {
            _.each(joinSequenseFilters, filter => {
              filter.meta = {
                version: 2
              };
              mainFilters.push(filter);
            });
          }
        }
      }
    }
    // end of dashboards360

    if (pinned) {
      mainFilters.push(..._.map(validateStateFilters(globalState), (f) => _.omit(f, ['$state', '$$hashKey'])));
    }

    // get the filters from the search meta
    if (metas && metas.savedDash && metas.savedDash.id !== dashboardId) {
      const msg = `Something wrong occurred, got dashboard=${dashboardId} but meta is from dashboard=${metas.savedDash.id}`;
      return Promise.resolve(new Error(msg));
    }
    const smFilters = metas && metas.savedSearchMeta && metas.savedSearchMeta.filter;
    if (smFilters) {
      _.each(smFilters, filter => {
        filter.meta.fromSavedSearch = true;
      });
      mainFilters.push(...smFilters);
    }
    // remove disabled filters
    if (!disabled) {
      mainFilters = _.filter(mainFilters, (f) => f.meta && !f.meta.disabled);
    }
    return Promise.resolve(uniqFilters(mainFilters, { state: true, negate: true, disabled: true }));
  };

  /**
   * Returns the current set of queries for the given dashboard
   */
  KibiState.prototype._getQueries = function (dashboardId, appState, metas) {
    let query = getDefaultQuery();
    if (appState && this.getCurrentDashboardId() === dashboardId) {
      if (appState.query) {
        query = _.cloneDeep(appState.query);
      }
    } else {
      const q = this._getDashboardProperty(dashboardId, this._properties.query);
      if (q) {
        query = _.cloneDeep(q);
      }
    }

    // get the query from the search meta
    if (metas && metas.savedDash && metas.savedDash.id !== dashboardId) {
      const msg = `Something wrong occurred, got dashboard=${dashboardId} but meta is from dashboard=${metas.savedDash.id}`;
      return Promise.resolve(new Error(msg));
    }
    const smQuery = metas && metas.savedSearchMeta && metas.savedSearchMeta.query;
    if (smQuery && !_.isEqual(smQuery, query) && !this._isDefaultQuery(smQuery)) {
      return Promise.resolve([ query, smQuery ]);
    }
    return Promise.resolve([ query ]);
  };


  KibiState.prototype._getTime = function (dashboardId) {
    const timeDefaults = config.get('timepicker:timeDefaults');
    const time = {
      mode: timeDefaults.mode,
      from: timeDefaults.from,
      to: timeDefaults.to
    };

    // NOTE:
    // here we have to check if there is a time in kibi state
    // and always prefer it as at when this method is called
    // the dashboard might not yet be fully loaded and the time from kibi_state not fully synced to global state
    const t = this._getDashboardProperty(dashboardId, this._properties.time);
    if (dashboardId === this.getCurrentDashboardId()) {
      if (t) {
        time.mode = t.m;
        time.from = t.f;
        time.to = t.t;
      } else if (timefilter && timefilter.enabled === true) {
        time.mode = timefilter.time.mode;
        time.from = timefilter.time.from;
        time.to = timefilter.time.to;
      } else {
        return null;
      }
    } else {
      const t = this._getDashboardProperty(dashboardId, this._properties.time);
      if (t) {
        time.mode = t.m;
        time.from = t.f;
        time.to = t.t;
      }
    }

    return time;
  };

  /**
   * Returns the current time for the given dashboard
   */
  KibiState.prototype._getTimeFilter = function (time, index) {
    if (!index || !time) {
      // do not reject - just return null
      // rejecting in this method would brake the Promise.all
      return Promise.resolve(null);
    }
    return this._turnTimeIntoFilter(time, index);
  };

  KibiState.prototype.turnTimeAndIndexPatternIntoFilter = function (time, indexPattern) {
    let filter = null;
    const timefield = indexPattern.timeFieldName && _.find(indexPattern.fields, { name: indexPattern.timeFieldName });

    if (timefield) {
      filter = {
        range : {
          [timefield.name]: {
            gte: parseWithPrecision(time.from, false, $rootScope.sirenTimePrecision).valueOf(),
            lte: parseWithPrecision(time.to, true, $rootScope.sirenTimePrecision).valueOf(),
            format: 'epoch_millis'
          }
        }
      };
    }

    return filter;
  };

  KibiState.prototype._turnTimeIntoFilter = function (time, indexId) {
    return indexPatterns.get(indexId)
      .then((indexPattern) => {
        return this.turnTimeAndIndexPatternIntoFilter(time, indexPattern);
      })
      .catch((error) => {
      // if the pattern does not match any index, do not break Promise.all and return a null filter.
        if (error instanceof IndexPatternMissingIndices) {
          return null;
        }
        throw error;
      });
  };

  /**
   * Taken from timefilter.getBounds
   */
  KibiState.prototype.getTimeBounds = function (dashboardId) {
    if (!dashboardId) {
      throw new Error('KibiState.getTimeBounds cannot be called with missing dashboard ID');
    }

    const timeDefaults = config.get('timepicker:timeDefaults');
    let timeFrom = timeDefaults.from;
    let timeTo = timeDefaults.to;

    if (dashboardId === this.getCurrentDashboardId()) {
      timeFrom = timefilter.time.from;
      timeTo = timefilter.time.to;
    } else {
      const t = this._getDashboardProperty(dashboardId, this._properties.time);
      if (t) {
        timeFrom = t.f;
        timeTo = t.t;
      }
    }

    return {
      min: parseWithPrecision(timeFrom, false, $rootScope.sirenTimePrecision),
      max: parseWithPrecision(timeTo, true, $rootScope.sirenTimePrecision)
    };
  };

  // In some places we are using the timeBasedIndices for multiple pairs of
  // indexPatternIds + dashboardIds
  // this helper method gets the list in the format
  // inputList: [
  //   {
  //     indexPatternId: id,
  //     dashboardIds: [id1, ...]
  //   },
  //   ..
  // ]
  //
  // and return the list of results of timeBasedIndices method calls
  //
  KibiState.prototype.timeBasedIndicesMap = function (inputList) {
    const promises = [];
    _.each(inputList, item => {
      promises.push(this.timeBasedIndices(item.indexPatternId, item.dashboardIds));
    });
    return Promise.all(promises)
      .then(res => {
        const outputList = _.cloneDeep(inputList);
        _.each(outputList, (item, index) => {
          item.timeBasedIndices = res[index];
        });
        return outputList;
      })
      .catch(notify.error);
  };

  /**
   * timeBasedIndices returns an array of time-expanded indices for the given pattern. The time range is the one taken from
   * the kibi state.
   *
   * @param indexPatternId the pattern to expand
   * @param dashboardIds the ids of dashboard to take a time-range from
   * @returns an array of indices name
   */
  KibiState.prototype.timeBasedIndices = function (indexPatternId, ...dashboardIds) {
    if (indexPatternId === null) {
      return Promise.resolve([]);
    }
    return indexPatterns.get(indexPatternId)
      .then((pattern) => {
        return pattern.toIndexList();
      })
      .catch((error) => {
      // If computing the indices failed because the pattern does not match any index return an empty list.
        if (error instanceof IndexPatternMissingIndices) {
          return [];
        }
        throw error;
      });
  };

  KibiState.prototype._readFromURL = function () {
    const stash = KibiState.Super.prototype._readFromURL.call(this);

    if (stash) {
      // check the join_sequence
      _.each(stash[this._properties.dashboards], (meta, dashboardId) => {
        this.disableFiltersIfOutdated(meta[this._properties.filters], dashboardId);
      });
    }
    return stash;
  };

  /**
   * Returns an array of dashboard IDs.
   * WARNING: this method returns only the ID of dashboards that have some state, e.g., some filters.
   */
  KibiState.prototype.getAllDashboardIDs = function () {
    return _.keys(this[this._properties.dashboards]);
  };

  const wrapPromise = function (p) {
    return new Promise(function (resolve, reject) {
      p.then(res => resolve(res)).catch(err => resolve(err));
    });
  };

  /**
   * Returns the current state of the dashboards with given IDs
   */
  KibiState.prototype.getStates = function (dashboardIds, suppressWarnings = false) {
    if (!(dashboardIds instanceof Array)) {
      return Promise.reject(new Error('Expected dashboardIds to be an Array'));
    }
    if (!dashboardIds.length) {
      return Promise.resolve({});
    }

    const options = {
      pinned: true,
      disabled: false
    };

    const appState = getAppState();
    const getMetas = this._getDashboardAndSavedSearchMetas(dashboardIds, false, suppressWarnings);

    return getMetas
      .then((metas) => {
        const promises = [];
        for (let i = 0; i < metas.length; i++) {
          const meta = metas[i];
          // this promises can not fail so we correctly report errors
          // lets wrap them

          const time = this._getTime(meta.savedDash.id);
          promises.push(meta.savedDash.id);
          promises.push(meta.savedSearchMeta ? meta.savedSearchMeta.index : null);
          promises.push(wrapPromise(this._getFilters(meta.savedDash.id, appState, meta, time, options)));
          promises.push(wrapPromise(this._getQueries(meta.savedDash.id, appState, meta)));
          promises.push(wrapPromise(this._getTimeFilter(time, meta.savedSearchMeta ? meta.savedSearchMeta.index : null)));
        }

        return Promise.all(promises)
          .then(results => {
            // create a map iterating every 5
            const statesMap = {};
            for (let i = 0; i < results.length; i = i + 5) {
              const dashId = results[i];
              const index = results[i + 1];
              const filters = results[i + 2];
              const queries = results[i + 3];
              const time = results[i + 4];

              if (filters instanceof Error) {
                statesMap[dashId] = { error: filters };
              } else if (queries instanceof Error) {
                statesMap[dashId] = { error: queries };
              } else if (time instanceof Error) {
                statesMap[dashId] = { error: time };
              } else {
                statesMap[dashId] = {
                  index,
                  filters,
                  queries,
                  time
                };
              }

              if (statesMap[dashId].error && !suppressWarnings) {
                notify.warning(statesMap[dashId].error);
              }
            }

            // here add the one for which the meta is missing
            _.each(dashboardIds, dashId => {
              if (!statesMap[dashId]) {
                statesMap[dashId] = {
                  error: new Error('Missing metadata')
                };
              }
            });

            return statesMap;
          });
      });
  };

  /**
   * Returns the current state of the dashboard with given ID
   */
  KibiState.prototype.getState = function (dashboardId) {
    if (!dashboardId) {
      return Promise.reject(new Error('Missing dashboard ID'));
    }

    const options = {
      pinned: true,
      disabled: false
    };

    const dashboardIds = [ dashboardId ];
    const appState = getAppState();

    if (!dashboardIds.length) {
      const msg = `Dashboards ${JSON.stringify(dashboardIds)} are not saved. It needs to be for one of the visualizations.`;
      return Promise.reject(new Error(msg));
    }

    // here ignore the missing meta as getState can be called
    // on a dashboard without associated savedSearch
    const getMetas = this._getDashboardAndSavedSearchMetas(dashboardIds);

    // check siren-federate plugin
    if (!this.isSirenJoinPluginInstalled()) {
      const error = 'The Siren Federate plugin is not installed. Please install the plugin and restart Elasticsearch.';
      return Promise.reject(new Error(error));
    }

    return getMetas
      .then((metas) => {
        const promises = [];

        // extra check for metas
        // if dashboardIds is empty or contains only 1 element
        //   - the meta can be missing
        // else
        //   - each dashboard must have corresponding meta as these mean that we are passing
        //   set of relationally connected dashboards
        if (dashboardIds.length > 1) {
          for (let i = 0; i < metas.length; i++) {
            if (!metas[i].savedSearchMeta) {
              const error = `The dashboard [${metas[i].savedDash.title}] is expected to be associated with a saved search.`;
              return Promise.reject(new Error(error));
            }
          }
        }

        for (let i = 0; i < metas.length; i++) {
          const meta = metas[i];
          const time = this._getTime(meta.savedDash.id);

          promises.push(this._getFilters(meta.savedDash.id, appState, meta, time, options));
          promises.push(this._getQueries(meta.savedDash.id, appState, meta));
          promises.push(this._getTimeFilter(time, meta.savedSearchMeta ? meta.savedSearchMeta.index : null));
        }
        return Promise.all(promises)
          .then(([ filters, queries, time, ...rest ]) => {
            const index = metas[0].savedSearchMeta ? metas[0].savedSearchMeta.index : null;
            return { index, filters, queries, time };
          });
      });
  };

  KibiState.prototype._reparseExtraProps = function (object) {
    const clone = _.cloneDeep(object);
    return JSON.parse(toJson(clone, angular.toJson));
  };

  /**
   * Saves the AppState to the KibiState
   */
  KibiState.prototype.saveAppState = function () {
    const currentDashboard = this.getDashboardOnView();
    const appState = getAppState();
    const options = {
      pinned: false,
      disabled: true,
      injectFwFilter: false
    };

    if (!appState || !currentDashboard) {
      return Promise.resolve(false);
    }
    const currentDashboardId = currentDashboard.id;
    return Promise.all([
      this._getFilters(currentDashboardId, appState, null, null, options),
      this._getQueries(currentDashboardId, appState, null),
      savedDashboards.find()
    ])
      .then(([ filters, queries, savedDashboardsRes ]) => {
        filters = this._reparseExtraProps(filters);
        queries = this._reparseExtraProps(queries);
        const savedDash = _.find(savedDashboardsRes.hits, (hit) => hit.id === currentDashboardId);
        if (!savedDash) {
          return Promise.reject(new Error(`Unable to get saved dashboard [${currentDashboardId}]`));
        }
        const meta = JSON.parse(savedDash.kibanaSavedObjectMeta.searchSourceJSON);
        const dashFilters = _.reject(meta.filter, this._isNotDashFilter);
        const dashQuery = _.find(meta.filter, (filter) => filter.query && filter.query.query_string && !filter.meta);
        if (!_.size(filters) && !_.size(dashFilters)) {
        // do not save filters
        // - if there are none; and
        // - if there are no filters but the dashboard is saved with some filters
          this._deleteDashboardProperty(currentDashboardId, this._properties.filters);
        } else {
          this._setDashboardProperty(currentDashboardId, this._properties.filters, filters);
        }
        // save the query
        // queries contains only one query, the one from appState, since the meta argument is null
        // in the call to _getQueries above.
        // The query from the appState is always equal to the wildcard query if nothing was entered in the search bar by the user.
        if (this._isDefaultQuery(queries[0]) && (!dashQuery || this._isDefaultQuery(dashQuery.query))) {
        // do not save the query:
        // - if it is the default query; and
        // - if the dashboard query is also the default one
          this._deleteDashboardProperty(currentDashboardId, this._properties.query);
        } else {
          this._setDashboardProperty(currentDashboardId, this._properties.query, queries[0]);
        }
        // save time
        if (this._isDefaultTime(timefilter.time.mode, timefilter.time.from, timefilter.time.to) &&
          (!savedDash.timeRestore || this._isDefaultTime(savedDash.timeMode, savedDash.timeFrom, savedDash.timeTo, true))) {
          this._deleteDashboardProperty(currentDashboardId, this._properties.time);
        } else {
          this._saveTimeForDashboardId(currentDashboardId, timefilter.time.mode, timefilter.time.from, timefilter.time.to);
        }
        this.save();
      });
  };

  KibiState.prototype.isSirenJoinPluginInstalled = function () {
    const plugins = elasticsearchPlugins.get();
    return !(plugins.indexOf('siren-federate') === -1 && plugins.indexOf('siren-vanguard') === -1);
  };

  KibiState.prototype.getDashboardById = function (dashboardId) {
    const stateDashboards = this[this._properties.dashboards];
    return (stateDashboards) ? this[this._properties.dashboards][dashboardId] : null;
  };

  KibiState.prototype.removeDashboardById = function (dashboardId) {
    let removed = false;
    if (this[this._properties.dashboards][dashboardId]) {
      delete this[this._properties.dashboards][dashboardId];
      removed = true;
    }

    if (this[this._properties.dashboards].length === 0) {
      delete this[this._properties.dashboards];
      removed = true;
    }

    if (removed) {
      removed = false;
      return this.save();
    }
  };

  KibiState.prototype.dragDashboardOnGraph = function ({ showGraphMsg = false, dashHasSearch = false } = {}) {
    this.emit('drag_on_graph', showGraphMsg, dashHasSearch);
  };

  KibiState.prototype.dropDashboardOnGraph = function (dashboardId) {
    this.emit('drop_on_graph', dashboardId);
  };

  return new KibiState();
}

uiRoutes
  .addSetupWork(kibiState => kibiState.init())
  .addSetupWork(elasticsearchVersion => elasticsearchVersion.init())
  .addSetupWork(elasticsearchPlugins => elasticsearchPlugins.init());

uiModules
  .get('kibana/kibi_state')
  .service('elasticsearchPlugins', (Promise, $http) => {
    let plugins;
    let pluginsWithVersions;

    return {
      init: _.once(function () {
        const requestPluginListFromES = $http.get(`${getBasePath()}/getElasticsearchPlugins`)
          .then(res => {
            plugins = res.data;
          });

        const requestPluginListFromESWithVersions = $http.get(`${getBasePath()}/getElasticsearchPlugins/versions`)
          .then(res => {
            pluginsWithVersions = res.data;
          });

        return Promise.all([
          requestPluginListFromES,
          requestPluginListFromESWithVersions
        ]);
      }),
      get(options) {
        if (options && options.version) {
          return pluginsWithVersions;
        } else {
          return plugins;
        }
      }
    };
  })
  .service('elasticsearchVersion', (Promise, $http) => {
    let version;
    let major;
    let minor;

    return {
      init: _.once(function () {
        return $http.get(`${getBasePath()}/elasticsearchVersion`)
          .then(res => {
            version = res.data[0];
            const params = version.split(/\./g);
            major = parseInt(params[0]);
            minor = parseInt(params[1]);
          });
      }),
      get() {
        return version;
      },
      getMajor() {
        return major;
      },
      getMinor() {
        return minor;
      }
    };
  })
  .service('kibiState', Private => Private(KibiStateProvider));
