import $ from 'jquery';
import _ from 'lodash';
import { VisRenderbotProvider } from 'ui/vis/renderbot';
import { VislibVisTypeBuildChartDataProvider } from 'ui/vislib_vis_type/build_chart_data';
import { FilterBarPushFilterProvider } from 'ui/filter_bar/push_filter';
import { KibanaMap } from './kibana_map';
import { GeohashLayer } from './geohash_layer';
import './lib/service_settings';
import './styles/_tilemap.less';
import { ResizeCheckerProvider } from 'ui/resize_checker';


module.exports = function MapsRenderbotFactory(Private, $injector, serviceSettings, Notifier, courier, getAppState) {

  const ResizeChecker = Private(ResizeCheckerProvider);
  const Renderbot = Private(VisRenderbotProvider);
  const buildChartData = Private(VislibVisTypeBuildChartDataProvider);
  const notify = new Notifier({ location: 'Coordinate Map' });

  class MapsRenderbot extends Renderbot {

    constructor(vis, $el, uiState) {
      super(vis, $el, uiState);
      this._buildChartData = buildChartData.bind(this);
      this._geohashLayer = null;
      this._kibanaMap = null;
      this._$container = $el;
      this._kibanaMapReady = this._makeKibanaMap($el);
      this._baseLayerDirty = true;
      this._dataDirty = true;
      this._paramsDirty = true;
      this._resizeChecker = new ResizeChecker($el);
      // kibi: no need for corresponding .off call as this._resizeChecker.destroy will take care about removing all listeners
      this._resizeChecker.on('resize', () => {
        if (this._kibanaMap) {
          this._kibanaMap.resize();
        }
      });
    }

    async _makeKibanaMap() {

      try {
        this._tmsService = await serviceSettings.getTMSService();
        this._tmsError = null;
      } catch (e) {
        this._tmsService = null;
        this._tmsError = e;
        notify.warning(e.message);
      }

      if (this._kibanaMap) {
        this._kibanaMap.destroy();
      }

      const containerElement = $(this._$container)[0];
      const options = _.clone(this._getMinMaxZoom());
      const uiState = this.vis.getUiState();
      const zoomFromUiState = parseInt(uiState.get('mapZoom'));
      const centerFromUIState = uiState.get('mapCenter');
      options.zoom = !isNaN(zoomFromUiState) ? zoomFromUiState : this.vis.type.params.defaults.mapZoom;
      options.center = centerFromUIState ? centerFromUIState : this.vis.type.params.defaults.mapCenter;

      this._kibanaMap = new KibanaMap(containerElement, options);
      this._kibanaMap.addDrawControl();
      this._kibanaMap.addFitControl();
      this._kibanaMap.addLegendControl();
      this._kibanaMap.persistUiStateForVisualization(this.vis);

      let previousPrecision = this._kibanaMap.getAutoPrecision();
      let precisionChange = false;
      this._kibanaMap.on('zoomchange', () => {
        precisionChange = (previousPrecision !== this._kibanaMap.getAutoPrecision());
        previousPrecision = this._kibanaMap.getAutoPrecision();
      });
      this._kibanaMap.on('zoomend', () => {

        const isAutoPrecision = _.get(this._chartData, 'geohashGridAgg.params.autoPrecision', true);
        if (!isAutoPrecision) {
          return;
        }

        this._dataDirty = true;
        if (precisionChange) {
          courier.fetch();
        } else {
          this._recreateGeohashLayer();
          this._dataDirty = false;
          this._doRenderComplete();
        }
      });


      this._kibanaMap.on('drawCreated:rectangle', event => {
        addSpatialFilter(_.get(this._chartData, 'geohashGridAgg'), 'geo_bounding_box', event.bounds, this.vis._siren);
      });
      this._kibanaMap.on('drawCreated:polygon', event => {
        addSpatialFilter(_.get(this._chartData, 'geohashGridAgg'), 'geo_polygon', { points: event.points }, this.vis._siren);
      });
      this._kibanaMap.on('baseLayer:loaded', () => {
        this._baseLayerDirty = false;
        this._doRenderComplete();
      });
      this._kibanaMap.on('baseLayer:loading', () => {
        this._baseLayerDirty = true;
      });
    }

    _getMinMaxZoom() {
      const mapParams = this._getMapsParams();
      if (this._tmsError) {
        return serviceSettings.getFallbackZoomSettings(mapParams.wms.enabled);
      } else {
        return this._tmsService.getMinMaxZoom(mapParams.wms.enabled);
      }
    }

    _recreateGeohashLayer() {

      if (this._geohashLayer) {
        this._kibanaMap.removeLayer(this._geohashLayer);
      }
      if (!this._chartData || !this._chartData.geoJson) {
        return;
      }
      const geohashOptions = this._getGeohashOptions();
      // kibi: destroy to correctly remove listeners
      if (this._geohashLayer) {
        this._geohashLayer.destroy();
      }
      // kibi: end
      this._geohashLayer = new GeohashLayer(this._chartData.geoJson, geohashOptions, this._kibanaMap.getZoomLevel(), this._kibanaMap);
      this._kibanaMap.addLayer(this._geohashLayer);
    }


    /**
     * called on data change
     * @param esResponse
     */
    render(esResponse) {
      this._dataDirty = true;
      this._kibanaMapReady.then(() => {
        this._chartData = this._buildChartData(esResponse);
        this._recreateGeohashLayer();
        this._kibanaMap.useUiStateFromVisualization(this.vis);
        this._kibanaMap.resize();
        this._dataDirty = false;
        this._doRenderComplete();
      });
    }

    destroy() {
      // kibi: destroy to correctly remove listeners
      if (this._geohashLayer) {
        this._geohashLayer.destroy();
      }
      // kibi: end
      if (this._kibanaMap) {
        this._kibanaMap.destroy();
      }
      this._resizeChecker.destroy();
    }

    setWmsLayer(mapParams, minZoom, maxZoom) {
      this._kibanaMap.setBaseLayer({
        baseLayerType: 'wms',
        options: {
          minZoom: minZoom,
          maxZoom: maxZoom,
          url: mapParams.wms.url,
          ...mapParams.wms.options
        }
      });
    }

    setTmsLayer() {
      if (!this._tmsError) {
        const url = this._tmsService.getUrl();
        const options = this._tmsService.getTMSOptions();
        this._kibanaMap.setBaseLayer({
          baseLayerType: 'tms',
          options: { url, ...options }
        });
      }
    }

    /**
     * called on options change (vis.params change)
     */
    async updateParams() {

      this._paramsDirty = true;
      await this._kibanaMapReady.then(() => {
        const mapParams = this._getMapsParams();
        const { minZoom, maxZoom } = this._getMinMaxZoom();

        if (mapParams.wms.enabled) {

          // kibi: improved if...else for correct rendering to set WMS layer only when
          // _makeKibanaMap() will be resolved and map will be ready
          if (maxZoom > this._kibanaMap.getMaxZoomLevel()) {
            // kibi: destroy to correctly remove listeners
            if (this._geohashLayer) {
              this._geohashLayer.destroy();
            }
            // kibi: end
            this._geohashLayer = null;
            this._kibanaMapReady = this._makeKibanaMap()
              .then(() => this.setWmsLayer(mapParams, minZoom, maxZoom));
          } else {
            this.setWmsLayer(mapParams, minZoom, maxZoom);
          }
          // kibi: end
        } else {

          // kibi: improved if...else for correct rendering to set TMS layer only when
          // _makeKibanaMap() will be resolved and map will be ready
          if (maxZoom < this._kibanaMap.getMaxZoomLevel()) {
            // kibi: destroy to correctly remove listeners
            if (this._geohashLayer) {
              this._geohashLayer.destroy();
            }
            // kibi: end
            this._geohashLayer = null;
            this._kibanaMapReady = this._makeKibanaMap()
              .then(() => {
                // kibi: removed _kibanaMap.setZoomLevel(maxZoom)
                // to keep consistent zoom after layer toggling
                this.setTmsLayer();
              });
          } else {
            this.setTmsLayer();
          }
          // kibi: end
        }
        const geohashOptions = this._getGeohashOptions();
        if (!this._geohashLayer || !this._geohashLayer.isReusable(geohashOptions)) {
          this._recreateGeohashLayer();
        }

        this._kibanaMap.setDesaturateBaseLayer(mapParams.isDesaturated);
        this._kibanaMap.setShowTooltip(mapParams.addTooltip);
        this._kibanaMap.setLegendPosition(mapParams.legendPosition);

        this._kibanaMap.useUiStateFromVisualization(this.vis);
        this._kibanaMap.resize();
        this._paramsDirty = false;
        this._doRenderComplete();
      });
    }

    _getMapsParams() {
      return _.assign(
        {},
        this.vis.type.params.defaults,
        { type: this.vis.type.name },
        this.vis.params
      );
    }

    _getGeohashOptions() {
      const newParams = this._getMapsParams();
      return {
        valueFormatter: this._chartData ? this._chartData.valueFormatter : null,
        tooltipFormatter: this._chartData ? this._chartData.tooltipFormatter : null,
        mapType: newParams.mapType,
        heatmap: {
          heatClusterSize: newParams.heatClusterSize
        }
      };
    }

    _doRenderComplete() {
      if (this._paramsDirty || this._dataDirty || this._baseLayerDirty) {
        return;
      }
      this.$el.trigger('renderComplete');
    }

  }

  function addSpatialFilter(agg, filterName, filterData, _siren) {
    if (!agg) {
      return;
    }

    const indexPatternName = agg.vis.indexPattern.id;
    const field = agg.fieldName();
    const filter = {};
    filter[filterName] = { ignore_unmapped: true };
    filter[filterName][field] = filterData;

    const putFilter = Private(FilterBarPushFilterProvider)(getAppState());
    return putFilter(filter, false, indexPatternName, _siren);
  }


  return MapsRenderbot;
};
