'use strict';

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _minimatch = require('minimatch');

var _minimatch2 = _interopRequireDefault(_minimatch);

var _ui_app_collection = require('./ui_app_collection');

var _ui_app_collection2 = _interopRequireDefault(_ui_app_collection);

var _ui_nav_link_collection = require('./ui_nav_link_collection');

var _ui_nav_link_collection2 = _interopRequireDefault(_ui_nav_link_collection);

var _ui_mappings = require('./ui_mappings');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class UiExports {
  constructor({ urlBasePath }) {
    this.navLinks = new _ui_nav_link_collection2.default(this);
    this.apps = new _ui_app_collection2.default(this);
    this.aliases = {};
    this.urlBasePath = urlBasePath;
    this.exportConsumer = _lodash2.default.memoize(this.exportConsumer);
    this.consumers = [];
    this.bundleProviders = [];
    this.defaultInjectedVars = {};
    this.injectedVarsReplacers = [];
    this.mappings = new _ui_mappings.MappingsCollection();
  }

  consumePlugin(plugin) {
    plugin.apps = new _ui_app_collection2.default(this);

    const types = _lodash2.default.keys(plugin.uiExportsSpecs);
    if (!types) return false;

    const unkown = _lodash2.default.reject(types, this.exportConsumer, this);
    if (unkown.length) {
      throw new Error('unknown export types ' + unkown.join(', ') + ' in plugin ' + plugin.id);
    }

    for (const consumer of this.consumers) {
      consumer.consumePlugin && consumer.consumePlugin(plugin);
    }

    types.forEach(type => {
      this.exportConsumer(type)(plugin, plugin.uiExportsSpecs[type]);
    });
  }

  addConsumer(consumer) {
    this.consumers.push(consumer);
  }

  addConsumerForType(typeToConsume, consumer) {
    this.consumers.push({
      exportConsumer(uiExportType) {
        if (uiExportType === typeToConsume) {
          return consumer;
        }
      }
    });
  }

  exportConsumer(type) {
    for (const consumer of this.consumers) {
      if (!consumer.exportConsumer) continue;
      const fn = consumer.exportConsumer(type);
      if (fn) return fn;
    }

    switch (type) {
      case 'app':
      case 'apps':
        return (plugin, specs) => {
          for (const spec of [].concat(specs || [])) {

            const app = this.apps.new(_lodash2.default.defaults({}, spec, {
              id: plugin.id,
              urlBasePath: this.urlBasePath
            }));

            plugin.extendInit((server, options) => {
              // eslint-disable-line no-loop-func
              const wrapped = app.getInjectedVars;
              app.getInjectedVars = () => wrapped.call(plugin, server, options);
            });

            plugin.apps.add(app);
          }
        };

      case 'link':
      case 'links':
        return (plugin, specs) => {
          for (const spec of [].concat(specs || [])) {
            this.navLinks.new(spec);
          }
        };

      case 'visTypes':
      case 'fieldFormats':
      case 'spyModes':
      case 'chromeNavControls':
      case 'navbarExtensions':
      case 'managementSections':
      case 'objectActions': // kibi: object actions
      case 'devTools':
      case 'docViews':
      case 'hacks':
        return (plugin, spec) => {
          this.aliases[type] = _lodash2.default.union(this.aliases[type] || [], spec);
        };

      case 'visTypeEnhancers':
        return (plugin, spec) => {
          //used for plugins that augment capabilities of an existing visualization
          this.aliases.visTypes = _lodash2.default.union(this.aliases.visTypes || [], spec);
        };

      case 'bundle':
        return (plugin, spec) => {
          this.bundleProviders.push(spec);
        };

      case 'aliases':
        return (plugin, specs) => {
          _lodash2.default.forOwn(specs, (spec, adhocType) => {
            this.aliases[adhocType] = _lodash2.default.union(this.aliases[adhocType] || [], spec);
          });
        };

      case 'injectDefaultVars':
        return (plugin, injector) => {
          plugin.extendInit(async (server, options) => {
            _lodash2.default.merge(this.defaultInjectedVars, (await injector.call(plugin, server, options)));
          });
        };

      case 'mappings':
        return (plugin, mappings) => {
          this.mappings.register(mappings, { plugin: plugin.id });
        };

      case 'replaceInjectedVars':
        return (plugin, replacer) => {
          this.injectedVarsReplacers.push(replacer);
        };
    }
  }

  find(patterns) {
    const aliases = this.aliases;
    const names = _lodash2.default.keys(aliases);
    const matcher = _lodash2.default.partialRight(_minimatch2.default.filter, { matchBase: true });

    return _lodash2.default.chain(patterns).map(function (pattern) {
      return names.filter(matcher(pattern));
    }).flattenDeep().reduce(function (found, name) {
      return found.concat(aliases[name]);
    }, []).value();
  }

  getAllApps() {
    const { apps } = this;
    return [...apps].concat(...apps.hidden);
  }

  getApp(id) {
    return this.apps.byId[id];
  }

  getHiddenApp(id) {
    return this.apps.hidden.byId[id];
  }

  getBundleProviders() {
    return this.bundleProviders;
  }
}

module.exports = UiExports;
