"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.nodeReferenceParser = nodeReferenceParser;
const unist_util_visit_1 = require("unist-util-visit");
/**
 * An AST parsing plugin for transforming node ids that we ask the LLM for format
 * (e.g. `[Google](siren_search_node:<entityId>|<id>)`) as a linkReference. This is then transformed into
 * custom nodes which we can render inside <NodeReferenceRenderer />.
 *
 * This plugin finds all link nodes whose URLs start with a certain prefix (e.g. `siren_search_node:`),
 * extracts the Node IDs and other information, and replaces the linkReference
 * with a custom node of type `nodeReferencePlugin`.
 *
 * @returns A plugin that processes an AST and transforms markdown links referencing nodes.
 * @see https://docs.siren.io/eui/#/editors-syntax/markdown-plugins on developing custom plugins
 */
function nodeReferenceParser() {
    return (tree) => {
        (0, unist_util_visit_1.visit)(tree, 'linkReference', (node, index, parent) => {
            const referenceLabel = node.label;
            const labelText = node.children?.[0]?.value;
            try {
                if (referenceLabel.startsWith('siren_search_node:')) {
                    const data = referenceLabel.slice('siren_search_node:'.length);
                    const [entityId, id] = data.split('|');
                    parent.children[index] = {
                        type: 'nodeReferencePlugin',
                        index: id.split('/_doc/')[0],
                        id: id.split('/_doc/')[1],
                        label: labelText || `${entityId}/${id}`,
                        entityId,
                        entityType: 'search'
                    };
                }
                else if (referenceLabel.startsWith('siren_eid_node:')) {
                    const data = referenceLabel.slice('siren_eid_node:'.length);
                    const [nodeId, eid] = data.split('|');
                    const entityId = eid;
                    parent.children[index] = {
                        type: 'nodeReferencePlugin',
                        id: nodeId,
                        label: labelText || eid,
                        entityId,
                        entityType: 'eid'
                    };
                }
                else if (referenceLabel.startsWith('siren_local_node:')) {
                    const data = referenceLabel.slice('siren_local_node:'.length);
                    const [uuid] = data.split('|');
                    parent.children[index] = {
                        type: 'nodeReferencePlugin',
                        id: uuid,
                        label: labelText || uuid,
                        isLocalNode: true
                    };
                }
            }
            catch (error) {
                // If the parsing fails, we log a warning and replace the node with a text node
                // which displays the node label instead.
                console.warn(`LLM was unable to format a node reference for: ${labelText}. \nLink reference: [${labelText}][${node.label}].`);
                parent.children[index] = {
                    type: 'text',
                    value: labelText,
                };
            }
        });
    };
}
