"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GenerativeAi = void 0;
const moment_1 = __importDefault(require("moment"));
const uuid_1 = require("uuid");
const axios_1 = require("axios");
const api = __importStar(require("../lib/api"));
const types_1 = require("../../common/types");
const serialize_graph_1 = require("../lib/serialize_graph");
const sessionContextId = (contextId) => `sirenapi-ai:${contextId}`;
function setContextItem(contextId, message) {
    sessionStorage.setItem(sessionContextId(contextId), JSON.stringify(message));
}
function getContextItem(contextId) {
    const item = sessionStorage.getItem(sessionContextId(contextId));
    return item ? JSON.parse(item) : [];
}
/**
 * Generative AI class. Accessible via `sirenapi.extensions.GenerativeAi`
 * @class
 */
class GenerativeAi {
    constructor() {
        throw new Error('Can not be instantiated');
    }
    /**
     * Send a prompt to the AI
     * @example await sirenapi.extensions.GenerativeAi.sendPrompt('What is 2 * 4?')
     * @param {string} message - The message to send to the AI
     * @param {object} [options] - Additional options
     * @param {string} [options.contextId] - The ID of the conversation context; helps the LLM remember previous messages
     * @returns {Promise<{ data: Array<K9Message | ToolCall>; contextId: string }>} - The response from the LLM
     */
    static async sendPrompt(message, options) {
        try {
            const contextId = options?.contextId || (0, uuid_1.v4)();
            let previousContext = getContextItem(contextId);
            const newMessage = { type: 'message', user: types_1.User.You, content: message, timestamp: (0, moment_1.default)().toISOString() };
            const currentMessages = [...previousContext, newMessage];
            const operation = api.sendMessages(currentMessages);
            const responseMessages = await operation.promise;
            previousContext = [...currentMessages, ...responseMessages];
            setContextItem(contextId, previousContext);
            return {
                data: responseMessages,
                contextId
            };
        }
        catch (e) {
            throw new Error(`Failed to get a response: ${e.message}`, { cause: e });
        }
    }
    /**
     * Generate a report from a graph model
     * @example
     * const graphVis = sirenapi.Visualization.getCurrentVisualizer()
     * const graphModel = await graphVis.getGraphModel()
     * await sirenapi.extensions.GenerativeAi.generateGraphReport(graphModel)
     *
     * @param {object} graphModel - The graph model to generate a report from. Retrieved from the `graphBrowserVis.getGraphModel()`
     * @param {object} [options] - Additional options
     * @param {string} [options.focusNodes] - The IDs of the nodes to focus on in the report
     * @param {string} [options.customPrompt] - The custom LLM prompt for generating the report
     * @returns {Promise<{ content: string }>} - The generated report content in Markdown format
     */
    static async generateGraphReport(graphModel, options) {
        try {
            const graph = await (0, serialize_graph_1.serializeGraphModel)(graphModel);
            return api.generateGraphReport(graph, options).promise;
        }
        catch (e) {
            const message = ((0, axios_1.isAxiosError)(e) && e.status === 504)
                ? 'A gateway timeout (504) occurred generating the graph report. Consider querying with a smaller graph or increasing the timeout.'
                : `Failed to generate graph report: ${e.message}`;
            throw new Error(message, { cause: e });
        }
    }
}
exports.GenerativeAi = GenerativeAi;
