"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AzureConfigSchema = void 0;
const joi_1 = __importDefault(require("joi"));
const openai_1 = require("openai");
const ai_client_1 = require("../ai_client");
const tools_1 = require("../tools");
class AzureOpenAiClient extends ai_client_1.AiClient {
    client;
    deploymentName;
    parameters;
    constructor(config) {
        super();
        this.client = new openai_1.AzureOpenAI({
            endpoint: config.connection.endpoint,
            apiKey: config.connection.apiKey,
            apiVersion: '2025-03-01-preview',
        });
        this.deploymentName = config.connection.deploymentName;
        this.parameters = config.parameters;
    }
    // The above API version unfortunately does not yet support OpenAI's new Responses API fully, so we need to keep
    // using the older Chat API for now. Keep an eye on it though, as it's a much better API to use.
    // https://learn.microsoft.com/en-us/azure/ai-services/openai/api-version-deprecation#latest-preview-api-releases
    async sendMessagesToAi(inputs, options) {
        const completion = await this.client.chat.completions.create({
            messages: convertLlmInputsToAzureAiInputs(inputs),
            top_p: this.parameters?.topP,
            temperature: this.parameters?.temperature,
            max_completion_tokens: this.parameters?.maxTokens,
            model: this.deploymentName, // Model is determined by the deployment
            tools: tools_1.tools.map(convertToolToChatApiTool),
            tool_choice: options?.disableTools ? 'none' : 'auto'
        });
        const outputs = [];
        if (completion.choices[0].message.content) {
            outputs.push({ content: completion.choices[0].message.content });
        }
        for (const toolCall of completion.choices[0].message.tool_calls || []) {
            outputs.push({
                type: 'function_call',
                callId: toolCall.id,
                function: toolCall.function
            });
        }
        return outputs;
    }
}
exports.default = AzureOpenAiClient;
/**
 * Converts LlmInputs where every message, tool call, and tool response is an individual array item into
 * the Azure OpenAI Chat API format, where a single message may contain a message and multiple tool calls
 */
function convertLlmInputsToAzureAiInputs(inputs) {
    const completionInputs = [];
    for (const input of inputs) {
        if (input.type === 'message') {
            completionInputs.push({ role: input.role, content: input.content });
        }
        else if (input.type === 'function_call') {
            const toolCall = { type: 'function', id: input.callId, function: input.function };
            const previousInput = completionInputs.at(-1);
            if (previousInput?.role === 'assistant') {
                previousInput.tool_calls = [...(previousInput.tool_calls || []), toolCall];
            }
            else {
                completionInputs.push({ role: 'assistant', tool_calls: [toolCall] });
            }
        }
        else {
            completionInputs.push({ role: 'tool', tool_call_id: input.callId, content: input.output });
        }
    }
    return completionInputs;
}
/** Converts a tool defined using the new Responses API schema to the older Chat API schema */
function convertToolToChatApiTool(tool) {
    return {
        type: 'function',
        function: {
            name: tool.name,
            description: tool.description || undefined,
            parameters: tool.parameters || undefined,
            strict: tool.strict
        }
    };
}
exports.AzureConfigSchema = joi_1.default.object({
    connection: joi_1.default.object({
        endpoint: joi_1.default.string().required(),
        deploymentName: joi_1.default.string().required(),
        apiKey: joi_1.default.string().required(),
    }).required(),
    parameters: joi_1.default.object({
        temperature: joi_1.default.number().min(0).max(1),
        topP: joi_1.default.number().min(0).max(1),
        maxTokens: joi_1.default.number().min(1),
    })
});
