"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.serializeGraphModel = serializeGraphModel;
// Disabled due to ESLint bug: https://github.com/import-js/eslint-plugin-import/issues/1810
// eslint-disable-next-line import/no-unresolved
const sync_1 = require("csv-stringify/browser/esm/sync");
const flat_1 = require("flat");
const dedent_1 = __importDefault(require("dedent"));
// @ts-ignore
const ng_injector_1 = require("ui/ng_injector");
// @ts-ignore
const saved_object_utils_1 = require("../../../../src/siren_core_plugins/federate_resolver/saved_objects/saved_object_utils");
const types_1 = require("../types");
async function serializeGraphModel(graph, selectedItemIds) {
    if (!graph.nodes.length) {
        return (0, dedent_1.default) `
      # Graph data

      There are no nodes or edges in this graph
    `;
    }
    const allEdgesWithoutChildren = graph.edges
        .map(edge => (0, types_1.isGroupedEdge)(edge) ? edge.children : edge)
        .flat();
    const [normalNodesSection, eidNodesSection, normalEdgesSection, localEdgesSection] = await Promise.all([
        generateNormalSearchNodesSection(graph.nodes.filter(types_1.isNormalSearchNode)),
        generateEidNodesSection(graph.nodes.filter(types_1.isEidNode)),
        generateNormalEdgesSection(allEdgesWithoutChildren.filter(types_1.isNormalEdge)),
        generateLocalEdgesSection(allEdgesWithoutChildren.filter(types_1.isLocalEdge))
    ]);
    return (0, dedent_1.default) `
    # Graph data

    Graphs consist of nodes and edges. The data that describes the graph below is structured in CSV.

    ${normalNodesSection}

    ${eidNodesSection}

    ${generateLocalNodesSection(graph.nodes.filter(types_1.isLocalNode))}

    ${normalEdgesSection}

    ${generateAggregatedEdgesSection(graph.edges.filter(types_1.isAggregatedEdge))}

    ${localEdgesSection}

    ${generateGroupedEdgesSection(graph.edges.filter(types_1.isGroupedEdge))}

    ${generateSelectedItemsSection(selectedItemIds)}

    ${generateGraphSummary(graph)}
  `;
}
async function generateNormalSearchNodesSection(nodes) {
    if (!nodes.length) {
        return '';
    }
    const savedSearches = (0, ng_injector_1.getNgService)('savedSearches');
    const savedSearchMap = await savedSearches.getMap();
    let section = (0, dedent_1.default) `
    ## Normal nodes

    All the nodes described in this section are part of the main dataset and represent stored data. They are all associated with entities.
    An entity is a type of 'thing' (car, person, incident, etc.). Entities have a predefined field schema, which is how node data is
    structured. The columns in the table below that represent these fields have a \`data.\` prefix.
    \n
  `;
    const nodesByEntityId = Object.groupBy(nodes, node => node.siren.entityId);
    for (const [entityId, nodesForEntity] of Object.entries(nodesByEntityId)) {
        const savedSearch = savedSearchMap[entityId];
        if (!savedSearch) {
            console.error(`Could not find entity with ID ${entityId}`);
            continue; // Skip nodes referencing a saved search that no longer exists
        }
        section += (0, dedent_1.default) `
      ### Nodes for entity ID '${entityId}'

      - Entity label: ${savedSearch.title}
      - Entity index pattern: ${savedSearch.indexPattern.pattern}

      \`\`\`
      ${convertNodesToCsv(nodesForEntity, savedSearch)}
      \`\`\`\n\n
   `;
    }
    return section;
}
function convertNodesToCsv(nodes, savedSearch) {
    const mainFieldNames = savedSearch.columns;
    const processedNodes = [];
    for (const node of nodes) {
        const docSource = {};
        for (const mainField of mainFieldNames) {
            if (node.siren.d[mainField] !== undefined) {
                docSource[mainField] = node.siren.d[mainField];
            }
        }
        processedNodes.push((0, flat_1.flatten)({
            id: node.id,
            label: node.style.label.value,
            data: docSource,
        }));
    }
    return (0, sync_1.stringify)(processedNodes, { header: true });
}
async function generateEidNodesSection(eidNodes) {
    if (!eidNodes.length) {
        return '';
    }
    const savedEids = (0, ng_injector_1.getNgService)('savedEids');
    const savedEidsMap = await savedEids.getMap();
    let section = (0, dedent_1.default) `
    ## EID nodes

    EID stands for "entity identifier". These nodes are not backed by stored documents but represent real-world entities that are inferred
    from the data. They appear when multiple normal nodes reference the same value—such as a city or category.
    \n
  `;
    const nodesByEntityId = Object.groupBy(eidNodes, node => node.siren.entityId);
    for (const [entityId, nodesForEntity] of Object.entries(nodesByEntityId)) {
        const entity = savedEidsMap[entityId];
        if (!entity) {
            console.error(`Could not find entity with ID ${entityId}`);
            continue; // Skip nodes referencing an EID entity that no longer exists
        }
        const eidNodeInfo = nodesForEntity
            .map(node => ({ id: node.id, label: node.style.label.value, value: node.siren.eidValue }));
        section += (0, dedent_1.default) `
      ### Nodes for entity ID '${entityId}'

      - Entity label: ${entity.title}

      \`\`\`
      ${(0, sync_1.stringify)(eidNodeInfo, { header: true })}
      \`\`\`\n\n
   `;
    }
    return section;
}
function generateLocalNodesSection(nodes) {
    if (!nodes.length) {
        return '';
    }
    const localNodes = nodes
        .map(node => ({
        id: node.id,
        label: node.style.label.value,
        entityId: node.siren.entityId
    }));
    return (0, dedent_1.default) `
    ## Local nodes

    These nodes are temporary visual elements with only a label and no structured data. They may still reference an entity ID, but are
    not stored in the entity’s index and contain only a display label. Local nodes are used to represent entities that are not part of
    the main dataset or to provide additional context.

    \`\`\`
    ${(0, sync_1.stringify)(localNodes, { header: true })}
    \`\`\`
  `;
}
async function generateNormalEdgesSection(edges) {
    if (!edges.length) {
        return '';
    }
    const savedRelations = (0, ng_injector_1.getNgService)('savedRelations');
    const savedRecordAsRelations = (0, ng_injector_1.getNgService)('savedRecordAsRelations');
    const [savedRelationsMap, savedRecordAsRelationsMap] = await Promise.all([savedRelations.getMap(), savedRecordAsRelations.getMap()]);
    const savedRelationsMapWithInverses = { ...savedRelationsMap };
    for (const relation of Object.values(savedRelationsMap)) {
        savedRelationsMapWithInverses[relation.inverseOf] = relation;
    }
    let section = (0, dedent_1.default) `
    ## Normal edges

    These edges in the graph have an associated 'relation'. A relation is a predefined type of connection that links two entities. An edge
    is an instance of a relation, specifically linking two specific nodes in the graph.
    \n
  `;
    const edgesByRelId = Object.groupBy(edges, edge => edge.siren.relId);
    for (const [relId, edgesForRelation] of Object.entries(edgesByRelId)) {
        const relEndpoints = getRelationEndpoints(relId, edgesForRelation[0].siren.relIdInv, savedRelationsMapWithInverses, savedRecordAsRelationsMap);
        if (!relEndpoints) {
            console.error(`Could not find relation endpoints for ID ${relId} (or ${edgesForRelation[0].siren.relIdInv})`);
            continue; // Skip nodes referencing a saved relation that no longer exists
        }
        const { relationLabel, relationInverseLabel } = getRelationLabels(relId, savedRelationsMapWithInverses, savedRecordAsRelationsMap, edgesForRelation[0]);
        if (!relationLabel) {
            console.error(`Could not find relation for ID ${relId} (or ${edgesForRelation[0].siren.relIdInv})`);
            continue;
        }
        section += (0, dedent_1.default) `
      ### Edges for relation ID '${relId}'

      - Relation label: ${relationLabel}
      - Relation inverse label: ${relationInverseLabel}
      - Source entity IDs: ${relEndpoints.sourceEntityIds.join(', ')}
      - Target entity IDs: ${relEndpoints.targetEntityIds.join(', ')}

      \`\`\`
      ${convertEdgesToCsv(edgesForRelation, relationLabel, relationInverseLabel)}
      \`\`\`\n\n
    `;
    }
    return section;
}
function getRelationEndpoints(relId, inverseRelId, savedRelationMapWithInverses, savedRecordAsRelationsMap) {
    if (isRecordAsRelationId(relId)) {
        // Edges in the graph can represent either the saved relation or its inverse, so we need to try both IDs.
        if (!(savedRecordAsRelationsMap[relId] || savedRecordAsRelationsMap[inverseRelId])) {
            return;
        }
        if (savedRecordAsRelationsMap[relId]) {
            const { incomingRelationIds, outgoingRelationIds, entityId } = savedRecordAsRelationsMap[relId];
            return {
                sourceEntityIds: getEntityIdsFromRaR(incomingRelationIds, savedRelationMapWithInverses, entityId),
                targetEntityIds: getEntityIdsFromRaR(outgoingRelationIds, savedRelationMapWithInverses, entityId)
            };
        }
        if (savedRecordAsRelationsMap[inverseRelId]) {
            const { incomingRelationIds, outgoingRelationIds, entityId } = savedRecordAsRelationsMap[inverseRelId];
            // Reverse outgoingRelationIds & incomingRelationIds as we found the relation by its inverse id
            return {
                sourceEntityIds: getEntityIdsFromRaR(outgoingRelationIds, savedRelationMapWithInverses, entityId),
                targetEntityIds: getEntityIdsFromRaR(incomingRelationIds, savedRelationMapWithInverses, entityId)
            };
        }
    }
    else {
        if (savedRelationMapWithInverses[relId]) {
            const { domainId, rangeId } = savedRelationMapWithInverses[relId];
            return { sourceEntityIds: [domainId], targetEntityIds: [rangeId] };
        }
        if (savedRelationMapWithInverses[inverseRelId]) {
            const { domainId, rangeId } = savedRelationMapWithInverses[inverseRelId];
            // Reverse domainId & rangeId as we found the relation by its inverse id
            return { sourceEntityIds: [rangeId], targetEntityIds: [domainId] };
        }
        return;
    }
}
function getEntityIdsFromRaR(relations, savedRelationMap, middleEntityId) {
    return relations
        .filter(id => savedRelationMap[id])
        .map(id => {
        const relation = savedRelationMap[id];
        return relation.domainId !== middleEntityId ? relation.domainId : relation.rangeId;
    });
}
function isRecordAsRelationId(id) {
    return saved_object_utils_1.SavedObjectUtils.retrieveTypeFromId(id) === 'record-as-relation';
}
// We don't try to get the relation saved objects because aggregated relations may not have `aggrEdgeRelationIds`, the label is
// dynamically generated based on the aggregation itself (so differs for each edge), and the LLM can infer the source/target entities.
function generateAggregatedEdgesSection(edges) {
    if (!edges.length) {
        return '';
    }
    // docCount and metricValue aren't used because we don't have metric and field info to make sense of it
    const aggregatedEdges = edges.map(edge => ({
        id: edge.id,
        label: edge.style.label.value,
        labelHover: edge.siren.labelHover,
        sourceNodeId: edge.source,
        targetNodeId: edge.target
    }));
    return (0, dedent_1.default) `
    ## Aggregated edges

    Aggregated edges represent indirect relationships that pass through one intermediary node. They are used to simplify the graph by
    summarizing known connections into a single edge, even when the underlying data model involves a multi-step path.

    For example, if a company is linked to an investment, and that investment is linked to an investor, an aggregated edge can directly
    connect the company and the investor. This helps users quickly see that a relationship exists — without needing to display or traverse
    the intermediate node (in this case, the investment).

    Aggregated edges do not replace the original detailed connections, but instead act as a higher-level, inferred link to aid
    interpretation and navigation.

    \`\`\`
    ${(0, sync_1.stringify)(aggregatedEdges, { header: true })}
    \`\`\`
  `;
}
async function generateLocalEdgesSection(edges) {
    if (!edges.length) {
        return '';
    }
    const localEdges = edges
        .map((edge) => ({
        id: edge.id,
        label: edge.style.label.value,
        sourceNodeId: edge.source,
        targetNodeId: edge.target
    }));
    return (0, dedent_1.default) `
    ## Local edges

    These edges are temporary visual elements that connect nodes, but are not associated to any predefined relation. They are used to
    visually represent connections that are not part of the main dataset or to provide additional context. The type of connection is
    described solely by the label of the edge.

    \`\`\`
    ${(0, sync_1.stringify)(localEdges, { header: true })}
    \`\`\`
  `;
}
function generateGroupedEdgesSection(edges) {
    if (!edges.length) {
        return '';
    }
    const groupedEdges = edges.map(edge => ({
        id: edge.id,
        label: edge.style.label.value,
        children: edge.children.map(edge => edge.id).join(',')
    }));
    return (0, dedent_1.default) `
    ## Grouped edges

    Grouped edges are visual elements that represent multiple underlying edges between the same source and target nodes. They help
    declutter the graph by consolidating several connections into a single edge, while still indicating the presence of multiple
    relationships. The individual edges that make up each grouped edge are defined in other sections but are referenced here as
    \`children\` delimited by commas.

    \`\`\`
    ${(0, sync_1.stringify)(groupedEdges, { header: true })}
    \`\`\`
  `;
}
function generateSelectedItemsSection(selectedItems) {
    if (!selectedItems || !selectedItems.length) {
        return '';
    }
    return (0, dedent_1.default) `
    ## Selected items
    
    Pay particular attention to these nodes and edges that are selected by the user in the graph:
    ${selectedItems.map(id => `\`${id}\``).join(', ')}
  `;
}
function convertEdgesToCsv(edges, relationLabel, relationInverseLabel) {
    const processedEdges = edges.map(edge => {
        const hasCustomLabel = edge.style.label.value !== relationLabel && edge.style.label.value !== relationInverseLabel;
        return {
            id: edge.id,
            sourceNodeId: edge.source,
            targetNodeId: edge.target,
            customLabel: hasCustomLabel ? edge.style.label.value : undefined
        };
    });
    return (0, sync_1.stringify)(processedEdges, { header: true });
}
// A summary helps LLMs (especially weaker ones) understand global structure quickly.
function generateGraphSummary(graph) {
    const uniqueEntityIds = new Set(graph.nodes
        .filter(node => node.siren.entityId)
        .map(node => node.siren.entityId));
    const allEdgesWithoutChildren = graph.edges
        .map(edge => (0, types_1.isGroupedEdge)(edge) ? edge.children : edge)
        .flat();
    const uniqueRelationIds = new Set(allEdgesWithoutChildren.filter(types_1.isNormalEdge).map(edge => edge.siren.relId));
    return (0, dedent_1.default) `
    ## Summary

    - Nodes: ${uniqueEntityIds.size} entity types, ${graph.nodes.length} total nodes
    - Edges: ${uniqueRelationIds.size} relation types, ${allEdgesWithoutChildren.length} total edges
  `;
}
function getRelationLabels(relId, savedRelationMapWithInverses, savedRecordAsRelationsMap, edge) {
    if (isRecordAsRelationId(relId)) {
        const rar = savedRecordAsRelationsMap[relId];
        if (rar) {
            return {
                relationLabel: rar.directLabel.type === 'label' ? rar.directLabel.value : edge.style.label.value,
                relationInverseLabel: rar.inverseLabel.type === 'label' ? rar.inverseLabel.value : edge.siren.invLabel || ''
            };
        }
    }
    else {
        const relation = savedRelationMapWithInverses[relId];
        if (relation) {
            return {
                relationLabel: relation.directLabel,
                relationInverseLabel: relation.inverseLabel
            };
        }
    }
    return {};
}
