"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenAiConfigSchema = void 0;
exports.convertLlmInputToOpenAiInput = convertLlmInputToOpenAiInput;
exports.convertOpenAiOutputToLlmOutput = convertOpenAiOutputToLlmOutput;
const joi_1 = __importDefault(require("joi"));
const openai_1 = require("openai");
const ai_client_1 = require("../ai_client");
const tools_1 = require("../tools");
class OpenAiClient extends ai_client_1.AiClient {
    client;
    parameters;
    constructor(config) {
        super();
        this.client = new openai_1.OpenAI({ organization: config.connection.orgId, apiKey: config.connection.apiKey });
        this.parameters = config.parameters;
    }
    async sendMessagesToAi(inputs, options) {
        const response = await this.client.responses.create({
            input: inputs.map(convertLlmInputToOpenAiInput),
            top_p: this.parameters?.topP,
            temperature: this.parameters?.temperature,
            model: this.parameters?.model || 'gpt-3.5-turbo',
            tools: tools_1.tools,
            tool_choice: options?.disableTools ? 'none' : 'auto',
            store: false
        });
        return response.output.map(convertOpenAiOutputToLlmOutput);
    }
}
exports.default = OpenAiClient;
function convertLlmInputToOpenAiInput(input) {
    if ('content' in input) {
        return { role: input.role, content: input.content };
    }
    else if (input.type === 'function_call') {
        return { type: 'function_call', call_id: input.callId, name: input.function.name, arguments: input.function.arguments };
    }
    else {
        return { type: 'function_call_output', call_id: input.callId, output: input.output };
    }
}
function convertOpenAiOutputToLlmOutput(llmOutput) {
    if (llmOutput.type === 'message') {
        const content = llmOutput.content[0];
        return { content: content.type === 'output_text' ? content.text : content.refusal };
    }
    else if (llmOutput.type === 'function_call') {
        return {
            type: 'function_call',
            callId: llmOutput.call_id,
            function: {
                name: llmOutput.name,
                arguments: llmOutput.arguments,
            }
        };
    }
    else {
        throw new Error(`Unexpected output type from LLM: ${llmOutput.type}`);
    }
}
exports.OpenAiConfigSchema = joi_1.default.object({
    connection: joi_1.default.object({
        orgId: joi_1.default.string().required(),
        apiKey: joi_1.default.string().required(),
    }).required(),
    parameters: joi_1.default.object({
        model: joi_1.default.string(),
        temperature: joi_1.default.number().min(0).max(1).default(0.5),
        topP: joi_1.default.number().min(0).max(1).default(1),
    })
});
