"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getChatRoute = getChatRoute;
const boom_1 = require("@hapi/boom");
const moment_1 = __importDefault(require("moment/moment"));
// @ts-ignore
const audited_route_handler_1 = require("../../../../src/siren_core_plugins/siren_audit/server/audited_route_handler");
// @ts-ignore
const constants_1 = require("../../../../src/siren_core_plugins/siren_audit/common/constants");
const types_1 = require("../../common/types");
const ai_client_1 = require("../ai_client");
const register_ui_contexts_1 = require("../setup/register_ui_contexts");
const route_utils_1 = require("./route_utils");
const chat_system_message_1 = require("./prompts/chat_system_message");
const graph_system_message_1 = require("./prompts/graph_system_message");
function getChatRoute(server) {
    const provider = server.config().get('siren-ai.provider');
    const config = server.config().get(`siren-ai.providerConfig.${provider}`);
    const ai = (0, ai_client_1.getAiClient)(provider, config);
    const { mockLlmResponses } = (0, route_utils_1.getDevConfig)(server);
    return {
        path: '/api/ai/chat',
        method: 'POST',
        options: {
            pre: [{ method: (0, route_utils_1.isRequestAllowed)(register_ui_contexts_1.SirenAiContexts.LLM_ACCESES_PERMISSION) }]
        },
        handler: (0, audited_route_handler_1.auditedRouteHandler)(constants_1.LOGGED_TYPES.LLMPrompt, async function (req) {
            const { messages, responseFormat, userSystemPrompt } = req.payload;
            if (!messages) {
                throw (0, boom_1.badRequest)('Body must include an "inputs" attribute');
            }
            if (mockLlmResponses) {
                return [{
                        type: 'message',
                        user: types_1.User.K9,
                        content: "Hello, I am a mocked response. Isn't that great!",
                        timestamp: (0, moment_1.default)().toISOString()
                    }];
            }
            const systemMessages = [await (0, chat_system_message_1.getChatSystemMessage)(responseFormat)];
            if (hasGraphAttachment(messages)) {
                systemMessages.push(await (0, graph_system_message_1.getGraphSystemMessage)());
            }
            if (userSystemPrompt) {
                systemMessages.push({ type: 'message', role: 'system', content: userSystemPrompt });
            }
            const inputs = [
                ...systemMessages,
                ...req.payload.messages.map(convertMessageToLlmInput)
            ];
            const llmOutputs = await (0, route_utils_1.sendMessagesToAiBoomifyErrors)(ai, inputs);
            return llmOutputs.map(convertLlmOutputToMessage);
        })
    };
}
function getAttachmentText(attachment) {
    return attachment?.model
        ? `\n Here is the serialized graph data relating to my above query:\n${attachment.model}`
        : '';
}
function convertMessageToLlmInput(message) {
    if (message.type === 'message') {
        return {
            type: 'message',
            role: message.user === types_1.User.K9 ? 'assistant' : 'user',
            content: `${message.content}${message.user === types_1.User.You ? getAttachmentText(message.attachment) : ''}`
        };
    }
    else {
        return message;
    }
}
function convertLlmOutputToMessage(llmOutput) {
    if ('callId' in llmOutput) {
        return llmOutput;
    }
    else {
        return { type: 'message', user: types_1.User.K9, content: llmOutput.content || '\<no response\\>', timestamp: (0, moment_1.default)().toISOString() };
    }
}
function hasGraphAttachment(messages) {
    return messages.some((message) => {
        return (message.type === 'message' &&
            message.user === types_1.User.You &&
            message.attachment?.model);
    });
}
