declare let React: any;
declare let Eui: Record<string, (...args: any[]) => any>;
declare let axios: {
  get: (url: string, config?: { responseType: 'blob' }) => Promise<{ data: Blob }>;
  isAxiosError: (error: any) => error is { status: number };
};

const {
  EuiMarkdownFormat,
  EuiButton,
  EuiCallOut,
  EuiHorizontalRule,
  EuiLoadingContent,
  getDefaultEuiMarkdownParsingPlugins,
} = Eui;

let reportTitle: string;

const markdownPlugins = getDefaultEuiMarkdownParsingPlugins();
for (const plugin of markdownPlugins) {
  if (plugin[1]?.emoticon) {
    plugin[1].emoticon = false;
  }
}

async function main(): Promise<void> {
  reportTitle = await getReportTitle();
  await currentVisualization.openFlyout({
    titleText: reportTitle,
    Element: <FlyoutBody/>,
    size: 's'
  });
}

function ReportCallout(): React.ReactElement {
  return (
    <EuiCallOut
      size="s"
      title="Generative AI report"
      iconType="iInCircle"
      style={{ marginBottom: '15px', padding: '6px' }}
      data-test-subj="generative-ai-warning"
    >
      This report was generated using generative AI. Please verify its accuracy before use.
    </EuiCallOut>
  );
}

function FlyoutBody(): React.ReactElement {
  const [content, setContent] = React.useState('');
  const [isLoading, setIsLoading] = React.useState(true);
  const [error, setError] = React.useState();
  const [graphImageUri, setGraphImageUri] = React.useState('');

  React.useEffect(() => {
    async function generateReportContent(): Promise<void> {
      if (!sirenapi.extensions.GenerativeAi) {
        setError('The Siren AI plugin is disabled or not installed.');
        return;
      }

      try {
        const [graph, selections] = await Promise.all([
          currentVisualization.getGraphModel(),
          currentVisualization.selection()
        ]);

        const selectedNodeIds = selections
          .filter(selection => !!selection.entityId)
          .map(node => node.id);

        const { content } = await sirenapi.extensions.GenerativeAi.generateGraphReport(graph, { focusNodes: selectedNodeIds });
        setContent(content);
      } catch (e: any) {
        setError(e.cause?.response?.data?.message || e.message);
      }
    }

    async function generateGraphImage(): Promise<void> {
      try {
        const graphImageDataUri = await getGraphImageDataUri();
        setGraphImageUri(graphImageDataUri);
      } catch (e: any) {
        sirenapi.notify.error(e.message);
      }
    }

    async function generateReport(): Promise<void> {
      setIsLoading(true);
      try {
        await Promise.all([generateReportContent(), generateGraphImage()]);
      } finally {
        setIsLoading(false);
      }
    }

    void generateReport();
  }, [setContent, setError, setIsLoading]);

  if (error) {
    return (
      <EuiCallOut data-test-subj="generative-ai-error" title="Something went wrong querying the LLM" color="danger" iconType="alert">
        {error}
      </EuiCallOut>
    );
  }

  return (
    <div>
      <ReportCallout />
      {isLoading ? <EuiLoadingContent lines={10} /> :
        <div data-test-subj="generated-content">
          <EuiMarkdownFormat parsingPluginList={markdownPlugins}>{content}</EuiMarkdownFormat>
        </div>
      }
      <div style={{
        position: 'sticky',
        bottom: 0,
        background: 'white',
        padding: '6px',
      }}>
        <EuiHorizontalRule size="full" margin="xs" />
        <EuiButton
          fill
          onClick={() => downloadReport(processReport(content, graphImageUri))}
          iconType="download"
          data-test-subj="download-report"
          disabled={isLoading}
        >
          Download Report
        </EuiButton>
      </div>
    </div>
  );
}

function processReport(markdown: string, graphImageDataUrl: string): string {
  if (graphImageDataUrl && graphImageDataUrl.length) {
    return `# ${reportTitle}\n\n![Graph](${graphImageDataUrl})\n\n${markdown}`;
  } else {
    return `# ${reportTitle}\n\n${markdown}`;
  }
}

async function downloadReport(markdown: string): Promise<void> {
  try {
    const reportBlob = await sirenapi.Reporting.markdownToDocxBlob(markdown);
    await sirenapi.Reporting.downloadBlob(reportBlob, 'Siren-Graph-Report.docx');
  } catch (e) {
    if (axios.isAxiosError(e) && e.status === 413) {
      sirenapi.notify.error('The report is too large to be downloaded. Please reduce the size of the graph and try again.');
    } else {
      sirenapi.notify.error(`Error preparing the report for download: ${(e as Error).message}`);
    }
  }
}

async function getGraphImageDataUri(): Promise<string> {
  try {
    const graphBlobUrl = await currentVisualization.exportGraphToBlobURL({ width: 3000 });
    const response = await axios.get(graphBlobUrl, { responseType: 'blob' });
    const graphBlob = response.data;

    // Convert the generated BLOB into a base64 data uri
    return new Promise((resolve, reject) => {
      const fr = new FileReader();
      fr.onload = () => resolve(fr.result as string);
      fr.onerror = () => reject(new Error('Failed to read the blob as a Data URI'));
      fr.readAsDataURL(graphBlob);
    });
  } catch (e: any) {
    throw new Error(`Error taking a snapshot of the Graph: ${e.message}`);
  }
}

async function getReportTitle(): Promise<string> {
  let graphName = '';
  try {
    graphName = (await currentVisualization.getOpenedGraph()).getLabel();
  } catch (e: any) {
    graphName = currentVisualization.title || 'Graph Browser';
  }

  return `Siren AI summary for ${graphName}`;
}

main()
  .catch(e => {
    console.error(e);
    sirenapi.notify.error(new Error(`An error occurred generating a report: ${e.message}`, { cause: e }));
  });
